//
// See the file LICENSE for redistribution information.
//
// Copyright (c) 2002-2005
//	Sleepycat Software.  All rights reserved.
//
// $Id: dbxmlsh.cpp,v 1.15 2005/04/19 14:13:17 gmf Exp $
//

#include <fstream>
#include <string>

#include "dbxml/DbXml.hpp"
#include "Environment.hpp"
#include "DefaultShell.hpp"
#ifdef HAVE_GETOPT
#include <unistd.h>
#else
extern "C" int getopt(int argc, char * const argv[], const char *optstring);
extern "C" char *optarg;
extern "C" int optind;
#endif

using namespace DbXml;
using namespace std;

static void usage(const string &progname, int exitCode)
{
	int pos = progname.rfind('/');
	if(pos == string::npos) {
		pos = progname.rfind('\\');
	}
	if(pos == string::npos) {
		pos = 0;
	}
	else {
		++pos;
	}

	cerr << "Usage: " << progname.substr(pos) << " [OPTIONS]" << endl << endl;
	cerr << "Options:" << endl;
	cerr << "-h <home>\tuse specified directory as a Berkeley DB environment." << endl;
	cerr << "-P <password>\tspecify database password" << endl;
	cerr << "-s <script>\trun the script in non-interactive mode.\n\t\t Multiple scripts are allowed" << endl;
	cerr << "-t\t\ttransaction mode -- transactions can be used, and are required \n\t\tfor writes" << endl;
	cerr << "-V\t\tprint software version" << endl;
	cerr << "-v\t\tverbose. Using the switch a second time increases verbosity" << endl;
	cerr << "-x\t\tsecure mode -- disallow access to the local file system and \n\t\tthe internet" << endl;
	cerr << "-?\t\thelp -- print this message" << endl;
	cerr << endl;
	exit(exitCode);
}

int main(int argc, char **argv)
{
	// Deal with command line arguments
	string path2DbEnv;
	u_int32_t joinEnvFlags =
		(DB_CREATE|DB_INIT_LOCK|DB_INIT_LOG|DB_INIT_MPOOL|DB_INIT_TXN|DB_USE_ENVIRON);
	u_int32_t envFlags = (DB_CREATE|DB_PRIVATE|DB_INIT_MPOOL);
	u_int32_t txnEnvFlags =	(DB_INIT_TXN|DB_INIT_LOCK|DB_INIT_LOG);
	u_int32_t dbxmlFlags = DBXML_ALLOW_EXTERNAL_ACCESS;
	vector<string> scripts;
	int verbose = 0;
	bool transactionMode = false;
	bool haveEnv = false;
	const char *progName = argv[0];
	const char *password = 0;
	int ch;

	while ((ch = getopt(argc, argv, "?h:hs:tvxVP:")) != -1) {
		switch (ch) {
		case 'h': {
			if(!path2DbEnv.length()) {
				path2DbEnv = optarg;
				haveEnv = true;
			}
			break;
		}
		case 'x': {
			dbxmlFlags &= ~DBXML_ALLOW_EXTERNAL_ACCESS;
			break;
		}
		case 't': {
			transactionMode = true;
			envFlags |= txnEnvFlags;
			break;
		}
		case 's': {
			scripts.push_back(optarg);
			break;
		}
		case 'v': {
			++verbose;
			break;
		}
		case 'V': {
			printf("%s\n", DbXml::dbxml_version(NULL, NULL, NULL));
			printf("%s\n", DbEnv::version(NULL, NULL, NULL));
			exit(0);
		}
		case 'P': {
			password = optarg;
			break;
		}
		case '?':
		default: {
			usage(progName, 0);
			break;
		}
		}
	}

	// Turn on logging if extra verbose is specified
	if(verbose > 1) {
		setLogLevel(LEVEL_ALL, true);
		setLogCategory(CATEGORY_ALL, true);
		setLogCategory(CATEGORY_NODESTORE, verbose > 2);
	}

	try {
		// Create a DB environment, and XmlManager
		DbEnv *dbenv = new DbEnv(0);
		if (password)
			dbenv->set_encrypt(password, DB_ENCRYPT_AES);
		dbenv->set_error_stream(&cerr);
		dbenv->set_cachesize(0, 64 * 1024 * 1024, 1);
		if (haveEnv) {
			int dberr = 0;
			try {
				if (verbose) {
					cout <<
					"Attempting to join environment: "
					     << path2DbEnv << endl;
				}
				dberr = dbenv->open(path2DbEnv.c_str(),
						    DB_JOINENV, 0);
				if (dberr == 0) {
					u_int32_t eflags = 0;
					dbenv->get_open_flags(&eflags);
					if (eflags & DB_INIT_TXN)
						transactionMode = true;
				}
			} catch (DbException &e) {
				if (e.get_errno() == DB_VERSION_MISMATCH) {
					cerr << "Error opening environment " <<
						path2DbEnv.c_str() << ": " <<
						e.what() << endl;
					exit(-1);
				}
				if (verbose)
					cerr << "Unable to join environment " <<
						path2DbEnv.c_str() <<
						", creating one" << endl; 
				// try again, with new flags
				// (no join), let it throw...
				dberr = dbenv->open(path2DbEnv.c_str(),
						    envFlags, 0);
			}
			if (dberr != 0) {
				cerr << "Error opening environment " <<
					path2DbEnv.c_str() << ", error is " <<
					dberr << endl;
				exit(-1);
			}
		} else
			dbenv->open(0, envFlags, 0);
		XmlManager db(dbenv, dbxmlFlags|DBXML_ADOPT_DBENV);

		// Create the environment
		Environment env(db);
		env.transactions() = transactionMode;

		// Create the Shell object
		DefaultShell shell;

		// Run scripts, if specified
		if(!scripts.empty()) {
			env.interactive() = false;
			env.verbose() = verbose != 0;

			for(vector<string>::iterator i = scripts.begin();
			    i != scripts.end() && !env.quit(); ++i) {
				ifstream scriptFile(i->c_str(), ios::in);
				if(!scriptFile) {
					cerr << progName << ": cannot open script file: " << *i << endl;
				} else {
					env.streamName() = *i;
					env.lineNo() = 0;
					shell.mainLoop(scriptFile, env);
					scriptFile.close();
				}
			}
		}

		// Perform the queries
		if(!env.quit()) {
			env.interactive() = true;
			env.verbose() = true;
			env.streamName() = "stdin";
			env.lineNo() = 0;
			shell.mainLoop(cin, env);
		}
	}
	catch(exception &e) {
		cerr << progName << ": error at lowest level: " << e.what() << endl;
		return 1;
	}
	catch(...) {
		cerr << progName << ": error at lowest level: " << endl;
		return 1;
	}

	return 0;
}
