//
// See the file LICENSE for redistribution information.
//
// Copyright (c) 2002-2005
//      Sleepycat Software.  All rights reserved.
//
// $Id: NsEventGenerator.cpp,v 1.24 2005/04/26 15:18:42 gmf Exp $
//

#include "NsEventGenerator.hpp"
#include "NsEvent.hpp"
#include "NsDocument.hpp"
#include "NsUtil.hpp"
#include "NsConstants.hpp"
#include "NsDocumentDatabase.hpp"
#include <pathan/XPath2MemoryManager.hpp>
#include <xercesc/framework/MemoryManager.hpp>

#define MAKEBOOL(x) ((x)? true : false)

// use 256K buffer for DB_MULTIPLE gets
#define NS_EVENT_DB_BUFSIZE (256 * 1024)

namespace DbXml {
/*
 * NsEventGeneratorBuf
 *
 * Helper class that encapsulates buffer mgmt for the DB bulk
 * retrieval API.  The only tricky part is knowing when all of the
 * DBTs in a multi-buffer are out of scope.  This occurs after child
 * processing is complete.
 */
class NsEventGeneratorBuf {
public:
	NsEventGeneratorBuf();
	// data
	NsEventGeneratorBuf *next;
	uint32_t nNodes;
	uint32_t nComplete;
	uint32_t nSize;
	bool usedUp;
	DbMultipleDataIterator iter;
};
	
}

using namespace DbXml;

//
// NsEventGenerator Implementation
//
/*
 * Iterate through a document, calling events
 *
 * Algorithm for a node:
 *   1.  if leading text, do lt.  in parent,
 *       while (current_text <= lt) text_event
 *   2.  startElem (with attrs)
 *   3.  If no children, spit out text events.  If child,
 *       do child.  (4) can be done by spitting out remaining
 *       text events.
 *   4.  if trailing text, do tt (could assert lastChild and
 *       that current_text of parent == tt index), spit out remaining
 *       text nodes.
 *   5.  next node:  if hasnext, do next, otherwise, return to parent
 *   state: parent, current_text index+ptr, next
 */

void
NsEventGenerator::_textEvent(const nsTextList_t *text,
			     uint32_t *currentText,
			     uint32_t lastIndex) const
{
	const nsTextEntry_t *entry;
	const xmlbyte_t *ptr;
	uint32_t ptrlen, type;
	NS_ASSERT(*currentText <= lastIndex)
	while (*currentText < lastIndex) {
		entry = &text->tl_text[*currentText];
		ptr = (const xmlbyte_t*)entry->te_text.t_chars;
		ptrlen = entry->te_text.t_len;
		type = entry->te_type;
		if (nsIsText(type) || nsIsCDATA(type)) {
			if (type & NS_IGNORABLE)
				_handler->ignorableWhitespace(ptr, ptrlen,
							      nsIsCDATA(type));
			else
				_handler->characters(ptr, ptrlen,
						     nsIsCDATA(type),
						     MAKEBOOL(type & NS_ENTITY_CHK));
		} else if (nsIsComment(type)) {
			_handler->comment(ptr, ptrlen);
		} else if (nsIsPI(type)) {
			uint32_t len = NsUtil::nsStringLen((xmlbyte_t*)ptr);
			const xmlbyte_t *data = ptr + len + 1;
			_handler->processingInstruction(ptr, data);
		} else if (nsIsEntityStart(type)) {
			_handler->startEntity(ptr, ptrlen);
		} else if (nsIsEntityEnd(type)) {
			_handler->endEntity(ptr, ptrlen);
		} else if (nsIsSubset(type)) {
			_handler->docTypeDecl(ptr, ptrlen);
		} else {
			NS_ASSERT(0) // bad text type
		}
		*currentText += 1;
	}
}

/*
 * Generate events for a node.
 * TBD: remove recursion.  Need to keep "previous" links
 * with nodes to do this.
 */
void
NsEventGenerator::_processNode(NsEventGeneratorBuf **curBuf,
			       const Dbt &data)
{
	// handle mid-stream stop request
	if (_stop)
		return;

	NsEventGeneratorBuf *mybuf = *curBuf;

	// Because the node buffer will stick around until we've finished,
	// we can set copyStrings to false, meaning that the nsNode_t strings
	// point directly into the node buffer.
	AutoDeallocate<nsNode_t>
		node(NsUtil::nsUnmarshalNode(_document.getMemoryManager(),
					     (unsigned char *)data.get_data(),
					     data.get_size(), /*copyStrings*/false),
		     _document.getMemoryManager());

	// element
	uint32_t flags = node->nd_header.nh_flags;
	bool isEmpty = !((flags & NS_HASCHILD) ||
			 (flags & NS_HASTEXT));
	const xmlbyte_t *name = (const xmlbyte_t*)nsName(node)->n_text.t_chars;
	const xmlbyte_t *prefix = _document.getPrefix8(nsNamePrefix(node));
	const xmlbyte_t *uri = nsHasUri(node) ? _document.getUri8(nsUriIndex(node)) : 0;
	// no event for doc node -- already done
	if (!(flags & NS_ISDOCUMENT)) {
		nsAttrList_t *attrs = node->nd_attrs;
		NsEventNodeAttrList8 alist(attrs, _document);
		_handler->startElement(name, prefix, uri, -1, &alist,
				       attrs ? attrs->al_nattrs : 0, isEmpty,
				       MAKEBOOL(flags & NS_ISROOT), node);
	}
	
	/* if child, do child; this will do ALL children */
	uint32_t curtext = 0;
	nsTextList_t *text = (flags & NS_HASTEXT) ? node->nd_text : 0;
	if (flags & NS_HASCHILD) {
		Dbt child;
		for(uint32_t i = 0; i < nsNumChildElem(node); ++i) {
			if(text) _textEvent(text, &curtext,
					    nsChildTextIndex(node,
							     i));
			_nextNode(curBuf, child);
			NS_ASSERT(child.get_data());
			_processNode(curBuf, child);
		}
	}
	
	/* done with this node.  Spit out remaining text */
	if (text)
		_textEvent(text, &curtext, nsNumText(node));

	if (!(flags & NS_ISDOCUMENT) && !isEmpty)
		_handler->endElement(name, prefix, uri,
				     MAKEBOOL(flags & NS_ISROOT),
				     node);

	_releaseNode(mybuf);
}

NsEventGenerator::NsEventGenerator(Transaction *txn, NsDocumentDatabase *db, const ID &docId,
				   u_int32_t flags, XER_NS MemoryManager *memManager,
				   const nid_t *startId)
	: _handler(0),
	  _document(memManager),
	  _stop(false),
	  _freeList(NULL)
{
	_document.initDoc(txn, db, docId, flags, false);

	docId.setDbtFromThis(_docKey);

	// set starting ID.  If NULL, start at beginning
	if (startId)
		_document.copyId(&_startId, startId);
	else {
		_startId.idLen = strlen(_docRootId) + 1;
		memcpy(_startId.idStore.idBytes, _docRootId, _startId.idLen);
	}
	// create a cursor and set it to the first data item
	db->getNodeStorageDatabase().getDb()
		.cursor(Transaction::toDbTxn(txn), &_cursor, flags);
}

NsEventGenerator::~NsEventGenerator()
{
	_cursor->close();
	while (_freeList) {
		NsEventGeneratorBuf *cur = _freeList;
		_freeList = cur->next;
		::free(cur);
	}
	_document.freeId(&_startId);
}

//virtual
void NsEventGenerator::start() {
	start(true);
}

void
NsEventGenerator::start(bool oneElementOnly)
{
	// first event is startDoc
	_handler->startDocument();
	// xmlDecl event only happens if the document has one
	if (_document.getXmlDecl8())
		_handler->xmlDecl(_document.getXmlDecl8(),
				 _document.getEncodingStr8(),
				 _document.getStandaloneStr());

	Dbt data;
	NsEventGeneratorBuf *buf = NULL;
	for(_nextNode(&buf, data, &_startId);
	    data.get_data() != 0 && !_stop;
	    _nextNode(&buf, data)) {
		_processNode(&buf, data);
		if (oneElementOnly) break;
	}
	// if buf is non-null, and not on free list, free it
	if (buf && !buf->next)
		::free(buf);

	_handler->endDocument();
}

void
NsEventGenerator::_releaseNode(NsEventGeneratorBuf *buf)
{
	buf->nComplete++;
	// it's not available until 1) all records have
	// been used up, and 2) all nodes it includes are
	// done processing
	if (buf->usedUp && (buf->nComplete == buf->nNodes)) {
		// done -- put it on free list
		buf->next = _freeList;
		_freeList = buf;
	}
}

//
// if startId is non-null, it's the first time, so
// do a GET vs NEXT cursor op.  *bufp will be NULL.
//
// No-content documents: it's possible to get here with a document
// ID that is not in the database, so silently handle such failures.
//
void
NsEventGenerator::_nextNode(NsEventGeneratorBuf **bufp, Dbt &data, nid_t *startId)
{
	NsEventGeneratorBuf *buf = *bufp;
	NS_ASSERT(buf || startId)
	uint32_t bufSize = NS_EVENT_DB_BUFSIZE;
	// are there any items left?
	if (buf) {
		if (!buf->iter.next(data)) {
			buf->usedUp = true;
			if (buf->nComplete == buf->nNodes) {
				buf->next = _freeList;
				_freeList = buf;
			}
			buf = 0;
		}
	}
	
	while (!buf) {
		// "construct" the object plus data
		Dbt data1; // does not need to live past this block
		uint32_t allocSize = bufSize + sizeof(NsEventGeneratorBuf);
		// pull off free list if available
		if (_freeList && (_freeList->nSize >= bufSize)) {
			buf = _freeList;
			_freeList = buf->next;
		} else {
			buf = (NsEventGeneratorBuf *) malloc(allocSize);
		}
		if (!buf) {
			NsUtil::nsThrowException(XmlException::NO_MEMORY_ERROR,
						 "Malloc failed",
						 __FILE__, __LINE__);
			return;
		}
		buf->next = 0;
		buf->nNodes = 0;
		buf->nComplete = 0;
		buf->usedUp = false;
		buf->nSize = bufSize;
		data1.set_flags(DB_DBT_USERMEM);
		data1.set_ulen(bufSize);
		data1.set_data(buf + 1);
		int ret = 0;
		try {
			if (startId) {
				xmlbyte_t *dptr = (xmlbyte_t *)
					data1.get_data();
				*dptr++ = NS_PROTOCOL_VERSION;
				memcpy(dptr, NsUtil::nsNode(startId),
				       startId->idLen);
				ret = _cursor->get(&_docKey,
					       &data1,
					       DB_GET_BOTH|DB_MULTIPLE);
			} else {
				ret = _cursor->get(&_docKey,
					       &data1, DB_NEXT|DB_MULTIPLE);
			}
		} catch (DbException &e) {
			// don't trust exception, check DBT to see
			// if not enough space
			if (data1.get_size() > data1.get_ulen()) {
				ret = ENOMEM;
			} else {
				NsUtil::nsThrowDbException(e, __FILE__, __LINE__);
			}
		}
		if (ret != 0) {
			::free(buf);
			if (ret == ENOMEM) {
				bufSize = data1.get_size() << 1; // 2x
				buf = 0;
				continue;
			} else if(startId) {
				// no-content document
				_stop = true;
				buf = 0;
			}
			*bufp = buf; // 0
			return;
		}
		// placement new to initialize multiple iter
		(void) new (&(buf->iter)) DbMultipleDataIterator(data1);
		if (!buf->iter.next(data)) {
			NsUtil::nsThrowException(XmlException::INTERNAL_ERROR,
						 "Failed to find node.",
						 __FILE__, __LINE__);
			return;
		}
		*bufp = buf;
	}
	buf->nNodes++;
}
