//
// See the file LICENSE for redistribution information.
//
// Copyright (c) 2002-2005
//      Sleepycat Software.  All rights reserved.
//
// $Id: NsDomEventSource.cpp,v 1.8 2005/04/05 16:44:06 bostic Exp $
//

#include "NsUtil.hpp"
#include "NsDomEventSource.hpp"
#include "NsDom.hpp"
#include "NsDocument.hpp"
#include "../UTF8.hpp"

using namespace DbXml;

void NsDomEventSource8::start() {
	NsDocument *doc = _startNode->getNsDocument();

	bool entityText = doc->getCreateEntityText();
	try {
		doc->setCreateEntityText(true);
		generateElement(_startNode);
	}
	catch(...) {
		doc->setCreateEntityText(entityText);
		throw;
	}
	doc->setCreateEntityText(entityText);
}

void NsDomEventSource8::generateElement(NsDomElement *element) const
{
	NsDocument *doc = element->getNsDocument();

	if(element->getNsNodeType() == nsNodeDocument) {
		_handler->startDocument();

		if(doc->getXmlDecl8() != 0 ||
		   doc->getEncodingStr8() != 0 ||
		   doc->getStandaloneStr() != 0) {
			_handler->xmlDecl(doc->getXmlDecl8(),
					  doc->getEncodingStr8(),
					  doc->getStandaloneStr());
		}

		generateElementChildren(element);

		_handler->endDocument();
	} else {
		nsNode_t *node = element->getNsNode();
		uint32_t attrcount = nsNumAttrs(node);
		bool utf16 = nsIsUTF16(node);

		const nsName_t *name = nsName(node);
		const xmlbyte_t *prefix = (name->n_prefix == NS_NOPREFIX ? 0 :
					   doc->getPrefix8(nsNamePrefix(node)));
		const xmlbyte_t *uri = (nsHasUri(node) ? doc->getUri8(nsUriIndex(node)) : 0);
		xmlbyte_t *localname = 0;
		if(utf16) NsUtil::nsToUTF8(doc->getMemoryManager(), &localname,
					   (const xmlch_t*)name->n_text.t_chars,
					   name->n_text.t_len + 1, 0);
		else localname = (xmlbyte_t*)name->n_text.t_chars;

		try {
			if(attrcount != 0) {
				NsEventNodeAttrList8 attrs(node->nd_attrs, *doc, nsIsUTF16(node));
				_handler->startElement(localname, prefix, uri, nsUriIndex(node),
						       &attrs, attrcount, !nsHasChildNode(node),
						       nsIsRoot(node), node);
			}
			else {
				_handler->startElement(localname, prefix, uri, nsUriIndex(node),
						       0, 0, !nsHasChildNode(node),
						       nsIsRoot(node), node);
			}

			if(nsHasChildNode(node)) {
				generateElementChildren(element);

				_handler->endElement(localname, prefix, uri, nsIsRoot(node), node);
			}
		}
		catch(...) {
			if(utf16) doc->getMemoryManager()->deallocate(localname);
			throw;
		}

		if(utf16) doc->getMemoryManager()->deallocate(localname);
	}
}

void NsDomEventSource8::generateElementChildren(NsDomElement *element) const
{
	NsDocument *doc = element->getNsDocument();

	NsDomNav *child = element->getNsFirstChild();
	while(child != 0) {
		switch(child->getNsNodeType()) {
		case nsNodeDocument:
		case nsNodeElement: {
			generateElement((NsDomElement*)child);
			break;
		}
		case nsNodeEntStart:
		case nsNodeEntEnd:
		case nsNodeText: {
			NsDomText *domtext = (NsDomText*)child;

			nsNode_t *node = domtext->getNsNode();
			bool utf16 = nsIsUTF16(node);
			nsTextEntry_t *tentry = nsTextEntry(node, domtext->getIndex());
			nsText_t *text = &(tentry->te_text);
			uint32_t textType = nsTextType(tentry->te_type);

			bool needsEscaping = nsTextEntityChk(tentry->te_type);
			xmlbyte_t *value = 0;
			int vlen = 0;
			if(utf16) {
				enum checkType ctype =
					(textType == NS_TEXT ? isCharacters :
					 ignore);
				vlen = NsUtil::nsToUTF8(doc->getMemoryManager(), &value,
							(const xmlch_t*)text->t_chars,
							text->t_len + 1, 0, &needsEscaping, ctype);
				vlen -= 1;
			} else {
				value = (xmlbyte_t*)text->t_chars;
				vlen = text->t_len;
			}

			try {
				switch(textType) {
				case NS_ENTSTART: {
					_handler->startEntity(value, vlen);
					break;
				}
				case NS_ENTEND: {
					_handler->endEntity(value, vlen);
					break;
				}
				case NS_CDATA: {
					_handler->characters(value, vlen,
							     true, false);
					break;
				}
				case NS_TEXT: {
					_handler->characters(value, vlen,
							     false, needsEscaping);
					break;
				}
				case NS_COMMENT: {
					_handler->comment(value, vlen);
					break;
				}
				case NS_PINST: {
					//
					// PI is encoded target0data0
					//
					const xmlbyte_t *data = value;
					while (*data++); // get past NULL
					_handler->processingInstruction(value, data);
					break;
				}
				case NS_SUBSET: {
					_handler->docTypeDecl(value, vlen);
					break;
				}
				default: NS_ASSERT(0);
				}
			}
			catch(...) {
				if(utf16) doc->getMemoryManager()->deallocate(value);
				throw;
			}

			if(utf16) doc->getMemoryManager()->deallocate(value);
			break;
		}
		default: NS_ASSERT(0);
		}

		child = child->getNsNextSibling();
	}
}
