//
// See the file LICENSE for redistribution information.
//
// Copyright (c) 2002-2005
//	Sleepycat Software.  All rights reserved.
//
// $Id: XmlValue.cpp,v 1.48 2005/04/20 18:31:29 bostic Exp $
//

#include "dbxml_config.h"
#include "dbxml/XmlPortability.hpp"
#include "dbxml/XmlValue.hpp"
#include "dbxml/XmlResults.hpp"
#include "dbxml/XmlData.hpp"
#include "Value.hpp"
#include <xercesc/dom/DOM.hpp>

#if defined(DBXML_DOM_XERCES2)
#if defined(XERCES_HAS_CPP_NAMESPACE)
  XERCES_CPP_NAMESPACE_USE
#endif
#endif

using namespace DbXml;

// function to test for whether the system has
// been initialized, which is triggered by active
// existence of at least one XmlManager object
static void
testInit()
{
	if (!Value::create()) {
	throw XmlException(
		XmlException::INVALID_VALUE, 
		"Cannot construct XmlValue instance without an active XmlManager object to initialize BDB XML");
	}
}

XmlValue::XmlValue()
	: value_(0)
{
	testInit();
	value_ = Value::create();
	value_->acquire();
}

XmlValue::XmlValue(Value *value)
	: value_(value)
{
	testInit();
	value_->acquire();
}

XmlValue::XmlValue(const std::string &v)
	: value_(0)
{
	testInit();
	value_ = new AtomicTypeValue(v);
	value_->acquire();
}

XmlValue::XmlValue(const char *v)
	: value_(0)
{
	testInit();
	value_ = new AtomicTypeValue(v);
	value_->acquire();
}

XmlValue::XmlValue(double v)
	: value_(0)
{
	testInit();
	value_ = new AtomicTypeValue(v);
	value_->acquire();
}

XmlValue::XmlValue(bool v)
	: value_(0)
{
	testInit();
	value_ = new AtomicTypeValue(v);
	value_->acquire();
}

XmlValue::XmlValue(const XmlDocument &document)
	:value_(0)
{
	testInit();
	value_ = new NodeValue(document);
	value_->acquire();
}

XmlValue::XmlValue(Type type, const std::string &v)
	: value_(0)
{
	testInit();
	// validate type
	value_ = Value::create(type, v, true);
	value_->acquire();
}

XmlValue::XmlValue(Type type, const XmlData &dbt)
	: value_(0)
{
	testInit();
	// validate type
	value_ = Value::create(type, dbt.getDbt(), true);
	value_->acquire();
}

XmlValue::~XmlValue()
{
	if(value_ != 0)
		value_->release();
}

XmlValue::XmlValue(const XmlValue &o)
	: value_(o.value_)
{
	value_->acquire();
}

XmlValue &XmlValue::operator = (const XmlValue &o)
{
	if (this != &o && value_ != o.value_) {
		if (value_ != 0)
			value_->release();
		value_ = o.value_;
		if (value_ != 0)
			value_->acquire();
	}
	return *this;
}

std::string XmlValue::getNodeName() const {
	if(isType(XmlValue::NODE)) {
		return ((NodeValue*)value_)->getNodeName();
	}
	throw XmlException(
			   XmlException::INVALID_VALUE, 
			   "Can't convert XmlValue to Node");
}

std::string XmlValue::getNamespaceURI() const {
	if(isType(XmlValue::NODE)) {
		return ((NodeValue*)value_)->getNamespaceURI();
	}
	throw XmlException(
			   XmlException::INVALID_VALUE, 
			   "Can't convert XmlValue to Node");
}

std::string XmlValue::getPrefix() const {
	if(isType(XmlValue::NODE)) {
		return ((NodeValue*)value_)->getPrefix();
	}
	throw XmlException(
			   XmlException::INVALID_VALUE, 
			   "Can't convert XmlValue to Node");
}

std::string XmlValue::getLocalName() const {
	if(isType(XmlValue::NODE)) {
		return ((NodeValue*)value_)->getLocalName();
	}
	throw XmlException(
			   XmlException::INVALID_VALUE, 
			   "Can't convert XmlValue to Node");
}

std::string XmlValue::getNodeValue() const {
	if(isType(XmlValue::NODE)) {
		return ((NodeValue*)value_)->getNodeValue();
	}
	throw XmlException(
			   XmlException::INVALID_VALUE, 
			   "Can't convert XmlValue to Node");

}

short XmlValue::getNodeType() const {
	if(isType(XmlValue::NODE)) {
		return ((NodeValue*)value_)->getNodeType();
	}
	throw XmlException(
			   XmlException::INVALID_VALUE, 
			   "Can't convert XmlValue to Node");

}

XmlValue XmlValue::getParentNode() const {
	if(isType(XmlValue::NODE)) {
		return ((NodeValue*)value_)->getParentNode();
	}
	throw XmlException(
			   XmlException::INVALID_VALUE, 
			   "Can't convert XmlValue to Node");

}

XmlValue XmlValue::getFirstChild() const {
	if(isType(XmlValue::NODE)) {
		return ((NodeValue*)value_)->getFirstChild();
	}
	throw XmlException(
			   XmlException::INVALID_VALUE, 
			   "Can't convert XmlValue to Node");

}

XmlValue XmlValue::getLastChild() const {
	if(isType(XmlValue::NODE)) {
		return ((NodeValue*)value_)->getLastChild();
	}
	throw XmlException(
			   XmlException::INVALID_VALUE, 
			   "Can't convert XmlValue to Node");

}

XmlValue XmlValue::getPreviousSibling() const {
	if(isType(XmlValue::NODE)) {
		return ((NodeValue*)value_)->getPreviousSibling();
	}
	throw XmlException(
			   XmlException::INVALID_VALUE, 
			   "Can't convert XmlValue to Node");

}

XmlValue XmlValue::getNextSibling() const {
	if(isType(XmlValue::NODE)) {
		return ((NodeValue*)value_)->getNextSibling();
	}
	throw XmlException(
			   XmlException::INVALID_VALUE, 
			   "Can't convert XmlValue to Node");

}

XmlResults XmlValue::getAttributes() const {
	if(isType(XmlValue::NODE)) {
		return ((NodeValue*)value_)->getAttributes();
	}
	throw XmlException(
			   XmlException::INVALID_VALUE, 
			   "Can't convert XmlValue to Node");

}

XmlValue XmlValue::getOwnerElement() const {
	if(isType(XmlValue::NODE)) {
		return ((NodeValue*)value_)->getOwnerElement();
	}
	throw XmlException(
			   XmlException::INVALID_VALUE, 
			   "Can't convert XmlValue to Node");

}

/// What type is the Value
XmlValue::Type XmlValue::getType() const
{
	return (value_->getType());
}

/// Does the value_ have no value_.
bool XmlValue::isNull() const
{
	return (value_ == Value::create());
}

bool XmlValue::isType(Type type) const
{
	return (value_->getType() == type);
}

bool XmlValue::isNumber() const
{
	return isType(XmlValue::DECIMAL) || isType(XmlValue::DOUBLE) ||
		isType(XmlValue::FLOAT);
}

bool XmlValue::isString() const
{
	return isType(XmlValue::STRING);
}

bool XmlValue::isBoolean() const
{
	return isType(XmlValue::BOOLEAN);
}

bool XmlValue::isNode() const
{
	return isType(XmlValue::NODE);
}

/// Return the value_ as a Number.
double XmlValue::asNumber() const
{
	return value_->asNumber();
}

/// Return the value_ as a String.
std::string XmlValue::asString() const
{
	return value_->asString();
}

/// Return the value_ as a String, using a non-default encoding
std::string XmlValue::asString(const std::string &encoding) const
{
	return value_->asString(encoding.c_str());
}

/// Return the value_ as a Boolean.
bool XmlValue::asBoolean() const
{
	return value_->asBoolean();
}

/// Return the value_ as a Node
XERCES_CPP_NAMESPACE_QUALIFIER DOMNode *XmlValue::asNode() const
{
	return value_->asNode();
}

/// Return the value_ as a Document.
const XmlDocument &XmlValue::asDocument() const
{
	return value_->asDocument();
}

/// Compare two value_s for equality.
bool XmlValue::equals(const XmlValue &v) const
{
	if (isNull()) {
		return v.isNull();
	} else {
		if (v.isNull()) {
			return false;
		} else {
			return value_->equals(*v.value_);
		}
	}
}

/// Compare two value_s for equality.
bool XmlValue::operator==(const XmlValue &v) const
{
	return equals(v);
}
