//
// See the file LICENSE for redistribution information.
//
// Copyright (c) 2002-2005
//	Sleepycat Software.  All rights reserved.
//
// $Id: Transaction.cpp,v 1.11 2005/04/05 16:44:03 bostic Exp $
//

#include "dbxml/XmlPortability.hpp"
#include "dbxml/XmlException.hpp"
#include "Transaction.hpp"
#include "Manager.hpp"
#include "Log.hpp"

using namespace DbXml;

extern "C" {
	static int abortFunction(DB_TXN *dbtxn);
	static int commitFunction(DB_TXN *dbtxn, u_int32_t flags);
};

// NOTE: Because a DbTxn is external, this constructor
// creates a hard reference on the
// object, which is only released by a commit or abort
// operation.  This allows the caller to commit or abort
// directly on the DbTxn object, even after a containing
// XmlTransaction object has been deleted.
Transaction::Transaction(XmlManager &mgr, DbTxn *txn)
	: txn_(txn), mgr_(mgr), owned_(false)
{
	// caller ensures txn is non-null
	setDbNotification(txn_->get_DB_TXN());
	acquire();
}

Transaction::Transaction(XmlManager &mgr, u_int32_t flags)
	: mgr_(mgr), owned_(true)
{
	try {
		int err = mgr_.getDbEnv()->txn_begin(0, &txn_, flags);
		if(err) {
			throw XmlException(err);
		}
		setDbNotification(txn_->get_DB_TXN());
	}
	catch(DbException &e) {
		throw XmlException(e);
	}
}

Transaction::Transaction(XmlManager &mgr, DbTxn *parent, u_int32_t flags)
	: mgr_(mgr), owned_(true)
{
	try {
		int err = mgr_.getDbEnv()->txn_begin(parent, &txn_, flags);
		if(err) {
			throw XmlException(err);
		}
		setDbNotification(txn_->get_DB_TXN());
	}
	catch(DbException &e) {
		throw XmlException(e);
	}
}

Transaction::~Transaction()
{
	// Make the transaction exception safe
	if(txn_) {
		abort();
	}
}

void Transaction::abort()
{
	if(!txn_) {
		throw XmlException(XmlException::TRANSACTION_ERROR,
				   "Cannot abort, transaction already committed or aborted");
	}
	try {
		int err = abortFunction(txn_->get_DB_TXN());
		if(err) {
			throw XmlException(err);
		}
	}
	catch(DbException &e) {
		throw XmlException(e);
	}
}

void Transaction::commit(u_int32_t flags)
{
	if(!txn_) {
		throw XmlException(XmlException::TRANSACTION_ERROR,
				   "Cannot commit, transaction already committed or aborted");
	}
	try {
		int err = commitFunction(txn_->get_DB_TXN(), flags);
		if(err) {
			throw XmlException(err);
		}
	}
	catch(DbException &e) {
		throw XmlException(e);
	}
}

Transaction *Transaction::createChild(u_int32_t flags)
{
	if(!txn_) {
		throw XmlException(XmlException::TRANSACTION_ERROR,
				   "Cannot create child, transaction already committed or aborted");
	}
	((Manager&)mgr_).checkFlags(Log::misc_flag_info,
				    "XmlTransaction::createChild()", flags,
				    DB_DIRTY_READ|DB_TXN_NOSYNC|DB_TXN_NOWAIT|
				    DB_TXN_SYNC|DB_DEGREE_2);
	return new Transaction(mgr_, txn_, flags);
}

DbTxn *Transaction::getDbTxn()
{
	if(!txn_) {
		throw XmlException(XmlException::TRANSACTION_ERROR,
				   "Cannot get DbTxn, transaction already committed or aborted");
	}
	return txn_;
}

void Transaction::preNotify(bool commit) const
{
	NotifyList::const_iterator end = notify_.end();
	NotifyList::const_iterator i = notify_.begin();
	for(; i != end; ++i) {
		(*i)->preNotify(commit);
	}
}

void Transaction::postNotify(bool commit) const
{
	NotifyList::const_iterator end = notify_.end();
	NotifyList::const_iterator i = notify_.begin();
	for(; i != end; ++i) {
		(*i)->postNotify(commit);
	}
}

void Transaction::unregisterNotify(Notify *notify)
{
	NotifyList::iterator end = notify_.end();
	NotifyList::iterator i = notify_.begin();
	while(i != end) {
		if(*i == notify) {
			i = notify_.erase(i);
			end = notify_.end();
		}
		else {
			++i;
		}
	}
}

void Transaction::setDbNotification(DB_TXN *txn)
{
	dbfuncs_.saved_abort_func = txn->abort;
	dbfuncs_.saved_commit_func = txn->commit;
	txn->abort = abortFunction;
	txn->commit = commitFunction;
	txn->xml_internal = (void *)this;
}

void Transaction::clearDbNotification(DB_TXN *txn)
{
	// caller will "release()" the reference
	txn->abort = dbfuncs_.saved_abort_func;
	txn->commit = dbfuncs_.saved_commit_func;
	txn->xml_internal = 0;
	txn_ = 0;
}

int Transaction::runDbNotification(DB_TXN *dbtxn, bool isCommit,
				   u_int32_t commitFlags)

{
	int err = 0;
	DbTxn *dbTxn = txn_;
	clearDbNotification(dbtxn); // nulls txn_
	preNotify(isCommit);
	// if owned_ call DbTxn method, which deletes the DbTxn
	if (isCommit) {
		if (owned_)
			err = dbTxn->commit(commitFlags);
		else
			err = dbfuncs_.saved_commit_func(dbtxn, commitFlags);
	} else {
		if (owned_)
			err = dbTxn->abort();
		else
			err = dbfuncs_.saved_abort_func(dbtxn);
	}
	postNotify(isCommit);
	return err;
}

extern "C" {
static int abortFunction(DB_TXN *txn)
{
	Transaction *xmlTxn = (Transaction *)txn->xml_internal;
	int err = 0;
	if (xmlTxn) {
		err = xmlTxn->runDbNotification(txn, false, 0);
		if (!xmlTxn->isOwned())
			xmlTxn->release();
	}
	return err;
}

static int commitFunction(DB_TXN *txn, u_int32_t flags)
{
	Transaction *xmlTxn = (Transaction *)txn->xml_internal;
	int err = 0;
	if (xmlTxn) {
		err = xmlTxn->runDbNotification(txn, true, flags);
		if (!xmlTxn->isOwned())
			xmlTxn->release();
	}
	return err;
}
}
