//
// See the file LICENSE for redistribution information.
//
// Copyright (c) 2002-2005
//	Sleepycat Software.  All rights reserved.
//
// $Id: SyntaxDatabase.hpp,v 1.10 2005/04/20 18:31:29 bostic Exp $
//

#ifndef __SYNTAXDATABASE_HPP
#define	__SYNTAXDATABASE_HPP

#include <string>
#include <vector>
#include "shared_ptr.hpp"
#include "DbWrapper.hpp"
#include "IDS.hpp"
#include "IndexEntry.hpp"

namespace DbXml
{

class KeyStatistics;
class Syntax;
class DbtIn;

/**
 * Wraps an index and statistics database for a syntax.
 * Container have one for each syntax available.
 */
class SyntaxDatabase
{
public:
	typedef shared_ptr<SyntaxDatabase> Ptr;
	typedef std::vector<Ptr> Vector;

	SyntaxDatabase(const Syntax *syntax, DbEnv *env, Transaction *txn, const std::string &name, u_int32_t pageSize, u_int32_t flags, int mode);
	~SyntaxDatabase();

	SecondaryDatabase *getIndexDB() { return index_.get(); }
	const SecondaryDatabase *getIndexDB() const { return index_.get(); }

	SecondaryDatabase *getStatisticsDB() { return statistics_.get(); }
	const SecondaryDatabase *getStatisticsDB() const { return statistics_.get(); }

	int getIndexData(OperationContext &context, IndexData::SharedPtr &data,
			 SecondaryDatabase::Operation op1, const Key &k1,
			 SecondaryDatabase::Operation op2 = SecondaryDatabase::NONE,
			 const Key &k2 = Key()) const;
	int getIDS(OperationContext &context, IDS::SharedPtr &ids,
		   SecondaryDatabase::Operation op1, const Key &k1,
		   SecondaryDatabase::Operation op2 = SecondaryDatabase::NONE,
		   const Key &k2 = Key()) const;

	void getStatistics(OperationContext &context, const Key &key,
			   KeyStatistics &es) const;
	int updateStatistics(OperationContext &context, DbtIn &key,
			     const KeyStatistics &statistics);

	void sync() {
		index_->sync();
		statistics_->sync();
	}
	static int dump(const Syntax *syntax, DbEnv *env,
			const std::string &name, std::ostream *out);
	static int load(const Syntax *syntax, DbEnv *env,
			const std::string &name, std::istream *in,
			unsigned long *lineno);
	static int verify(const Syntax *syntax, DbEnv *env,
			  const std::string &name, std::ostream *out,
			  u_int32_t flags);

private:
	// no need for copy and assignment
	SyntaxDatabase(const SyntaxDatabase&);
	SyntaxDatabase &operator=(const SyntaxDatabase&);

	const Syntax *syntax_;
	DbEnv *environment_;
	std::string containerName_;

	SecondaryDatabase::Ptr index_;
	SecondaryDatabase::Ptr statistics_;
};

}

#endif

