//
// See the file LICENSE for redistribution information.
//
// Copyright (c) 2002-2005
//Sleepycat Software.  All rights reserved.
//
// $Id: SyntaxDatabase.cpp,v 1.12 2005/04/05 16:44:03 bostic Exp $
//

#include "dbxml/XmlPortability.hpp"
#include "dbxml/XmlException.hpp"
#include "SyntaxDatabase.hpp"

#include <assert.h>
#include <sstream>
#include "Syntax.hpp"
#include "Key.hpp"
#include "OperationContext.hpp"
#include "KeyStatistics.hpp"
#include "Cursor.hpp"
#include "Container.hpp"
#include "IndexEntry.hpp"

using namespace DbXml;
using namespace std;

static const string index_name("document_index_");
static const string statistics_name("document_statistics_");

SyntaxDatabase::SyntaxDatabase(const Syntax *syntax, DbEnv *env, Transaction *txn, const std::string &name,
                               u_int32_t pageSize, u_int32_t flags, int mode)
	: syntax_(syntax),
	  environment_(env),
	  containerName_(name),
	  index_(new SecondaryDatabase(env, name, index_name + syntax->getName(), syntax, pageSize, flags & DB_XA_CREATE)),
	  statistics_(new SecondaryDatabase(env, name, statistics_name + syntax->getName(), syntax, pageSize, flags & DB_XA_CREATE))
{
	flags &= ~DB_XA_CREATE;
	int err = 0;
	try {

		// Open the Db objects
		err = index_->open(txn, true, flags, mode);
		if(err == 0)
			err = statistics_->open(txn, false, flags, mode);

	} catch (DbException &e) {
		if(e.get_errno() == EEXIST) {
			throw XmlException(XmlException::CONTAINER_EXISTS, e.what());
		}
		else {
			throw XmlException(e);
		}
	}
	if(err == EEXIST) {
		throw XmlException(XmlException::CONTAINER_EXISTS, db_strerror(err));
	}
	else if(err != 0) {
		throw XmlException(err);
	}
}

SyntaxDatabase::~SyntaxDatabase()
{
	// nothing to do
}

int SyntaxDatabase::getIndexData(OperationContext &context, IndexData::SharedPtr &data,
				 SecondaryDatabase::Operation op1, const Key &k1,
				 SecondaryDatabase::Operation op2, const Key &k2) const
{
	assert(k1.getSyntaxType() == syntax_->getType());
	assert(op2 == DbWrapper::NONE || k2.getSyntaxType() == syntax_->getType());

	scoped_ptr<SecondaryCursor> cursor;
	int err;
	if(op2 == DbWrapper::NONE) {
		err = const_cast<SecondaryDatabase*>(getIndexDB())->createCursor(context.txn(), cursor, op1, &k1);
	}
	else {
		err = const_cast<SecondaryDatabase*>(getIndexDB())->createCursor(context.txn(), cursor, op1, &k1, op2, &k2);
	}

	if(err == 0) {
		//
		// jcm - perf - Could we use the DB bulk get() instead here? Maybe just for equality?
		//
		scoped_ptr<IndexEntry> ie(new IndexEntry());
		err = cursor->first(*ie); // could throw
		while(err == 0 && ie->getDocID() != 0) {
			if(data->insert(ie.get()).second) {
				ie.release();
				ie.reset(new IndexEntry());
			}
			err = cursor->next(*ie); // could throw
		}
	}

	return err;
}

int SyntaxDatabase::getIDS(OperationContext &context, IDS::SharedPtr &ids,
			   SecondaryDatabase::Operation op1, const Key &k1,
			   SecondaryDatabase::Operation op2, const Key &k2) const
{
	IndexData::SharedPtr data(new IndexData);
	int err = getIndexData(context, data, op1, k1, op2, k2);
	if(err == 0) {
		uint32_t id;
		uint32_t lastid = 0;
		IndexData::iterator end = data->end();
		for(IndexData::iterator i = data->begin(); i != end; ++i) {
			id = (*i)->getDocID();
			if(id != lastid) {
				ids->push_back(id);
				lastid = id;
			}
		}
	}
	return err;
}

void SyntaxDatabase::getStatistics(OperationContext &context, const Key &key, KeyStatistics &es) const
{
	assert(key.getSyntaxType() == syntax_->getType());

	key.setDbtFromThis(context.key());
	int structureLength = syntax_->structureKeyLength(key.getIndex(), context.key().get_size());
	if(structureLength < (int)context.key().get_size()) {
		context.key().set_size(structureLength); // trim the value off
	}

	int err = statistics_->get(context.txn(), &context.key(), &context.data(), /*no flags*/0); // could throw on error
	if (err == 0) {
		es.setThisFromDbt(context.data());
	} else {
		es.zero();
	}
}

int SyntaxDatabase::updateStatistics(OperationContext &context, DbtIn &key, const KeyStatistics &statistics)
{
	Cursor cursor(environment_, statistics_->getDb(), context.txn(), CURSOR_WRITE);
	int err = cursor.get(&key, &context.data(), DB_SET | (context.txn() ? DB_RMW : 0)); // could throw on error
	if (err == 0) {
		KeyStatistics existing;
		existing.setThisFromDbt(context.data());
		existing.add(statistics);
		existing.setDbtFromThis(context.data());
		err = cursor.put(&key, &context.data(), DB_CURRENT); // could throw on error
	} else if (err == DB_NOTFOUND) {
		statistics.setDbtFromThis(context.data());
		err = cursor.put(&key, &context.data(), DB_KEYFIRST); // could throw on error
	}
	return err;
}

int SyntaxDatabase::dump(const Syntax *syntax, DbEnv *env, const std::string &name, std::ostream *out)
{
	SecondaryDatabase::Ptr index(new SecondaryDatabase(env, name, index_name + syntax->getName(), syntax, 0, 0));
	SecondaryDatabase::Ptr statistics(new SecondaryDatabase(env, name, statistics_name + syntax->getName(), syntax, 0, 0));

	int err = Container::writeHeader(index->getDatabaseName(), out);
	if(err == 0) err = index->dump(out);

	if(err == 0) err = Container::writeHeader(statistics->getDatabaseName(), out);
	if(err == 0) err = statistics->dump(out);

	return err;
}

int SyntaxDatabase::load(const Syntax *syntax, DbEnv *env, const std::string &name, std::istream *in, unsigned long *lineno)
{
	SecondaryDatabase::Ptr index(new SecondaryDatabase(env, name, index_name + syntax->getName(), syntax, 0, 0));
	SecondaryDatabase::Ptr statistics(new SecondaryDatabase(env, name, statistics_name + syntax->getName(), syntax, 0, 0));

	// Load index
	int err = Container::verifyHeader(index->getDatabaseName(), in);
	if(err != 0) {
		ostringstream oss;
		oss << "SyntaxDatabase::load() invalid database dump file loading '" << name << "'";
		Log::log(env, Log::C_DICTIONARY, Log::L_ERROR, oss.str().c_str());
	}
	else {
		err = index->load(in, lineno);
	}

	// Load statistics
	if(err == 0) {
		err = Container::verifyHeader(statistics->getDatabaseName(), in);
		if(err != 0) {
			ostringstream oss;
			oss << "SyntaxDatabase::load() invalid database dump file loading '" << name << "'";
			Log::log(env, Log::C_DICTIONARY, Log::L_ERROR, oss.str().c_str());
		}
		else {
			err = statistics->load(in, lineno);
		}
	}

	return err;
}

int SyntaxDatabase::verify(const Syntax *syntax, DbEnv *env, const std::string &name,
			   std::ostream *out, u_int32_t flags)
{
	SecondaryDatabase::Ptr index(new SecondaryDatabase(
					     env, name,
					     index_name + syntax->getName(),
					     syntax, 0, 0));
	// not all index DBs may exist, handle this by
	// attempting to open
	try {
		index->open(0, true, 0, 0);
	} catch (DbException &de) {
		if (de.get_errno() == ENOENT)
			return 0; // doesn't exist, nothing to do
		throw;
	}
	// open worked -- DB exists.  Now close it for verify.
	index.reset(new SecondaryDatabase(
			    env, name,
			    index_name + syntax->getName(),
			    syntax, 0, 0));
	SecondaryDatabase::Ptr statistics(new SecondaryDatabase(
						  env, name,
						  statistics_name + syntax->getName(),
						  syntax, 0, 0));
	int err = 0;
	if(flags & DB_SALVAGE) err = Container::writeHeader(
		index->getDatabaseName(), out);
	if(err == 0)
		err = index->verify(out, flags);
	
	if(err == 0 && (flags & DB_SALVAGE))
		err = Container::writeHeader(statistics->getDatabaseName(), out);
	if(err == 0)
		err = statistics->verify(out, flags);
	return err;
}
