//
// See the file LICENSE for redistribution information.
//
// Copyright (c) 2002-2005
//	Sleepycat Software.  All rights reserved.
//
// $Id: Statistics.hpp,v 1.29 2005/04/20 18:31:28 bostic Exp $
//

#ifndef __STATISTICS_HPP
#define	__STATISTICS_HPP

#include <map>
#include <vector>
#include "ID.hpp"
#include "KeyStatistics.hpp"
#include "ScopedDbt.hpp"
#include "Key.hpp"
#include "ReferenceCounted.hpp"

namespace DbXml
{

class Transaction;
class Container;
class Index;
class Syntax;
class OperationContext;

class Statistics : public ReferenceCounted
{
public:
	Statistics(const Container &container, Transaction *txn,const Index &index,
		   const char *child, const char *parent, const XmlValue &value);

	double getNumberOfIndexedKeys() const
	{
		return numIndexedKeys_;
	}
	double getNumberOfUniqueKeys() const
	{
		return numUniqueKeys_;
	}
	double getSumKeyValueSize() const
	{
		return sumKeyValueSize_;
	}

private:
	Statistics(const Statistics &);
	Statistics &operator=(const Statistics &);

	double numIndexedKeys_;
	double numUniqueKeys_;
	double sumKeyValueSize_;
};

/** The indexer makes use of this class to maintain statistics about
    the keys being generated. */
class StatisticsWriteCache
{
public:
	StatisticsWriteCache();
	~StatisticsWriteCache();
	// Reset the statistics object so that it can be reused.
	void reset();

	/// Add a key to the statistics. We assume that key has the correct endianness.
	void addToKeyStatistics(const Index &index, const Dbt &key, const Dbt &data, bool unique);

	/// Write the statistics to the statistics databases
	int updateContainer(OperationContext &context, Container &container) const;

private:
	StatisticsWriteCache(const StatisticsWriteCache&);
	StatisticsWriteCache &operator=(const StatisticsWriteCache &);

	typedef std::map<DbtIn, KeyStatistics> Dbt2KSMap;
	typedef std::vector<Dbt2KSMap*> Dbt2KSMapVector;
	Dbt2KSMapVector dkv_;
};

class StatisticsReadCache
{
public:
	StatisticsReadCache(const Container &container);

	void reset();

	const KeyStatistics &getKeyStatistics(OperationContext &context, const Key &key);
	double getPercentage(OperationContext &context, SecondaryDatabase::Operation op1, Key k1,
			     SecondaryDatabase::Operation op2 = SecondaryDatabase::NONE, Key k2 = Key());

private:
	struct PercentageKey {
		PercentageKey(SecondaryDatabase::Operation op1, Key k1, SecondaryDatabase::Operation op2, Key k2)
			: operation1(op1), operation2(op2), key1(k1), key2(k2) {}
		bool operator<(const PercentageKey &o) const;

		SecondaryDatabase::Operation operation1, operation2;
		Key key1, key2;
	};

	void populateStatistics(OperationContext &context, const Key &key);
	void putKeyStatistics(const Key &key, const KeyStatistics &stats);
	void putPercentage(const PercentageKey &pk, const double &p);

	typedef std::map<Key, KeyStatistics> StatsMap;
	typedef std::map<PercentageKey, double> PercentageMap;

	StatsMap statsMap_;
	PercentageMap percentageMap_;

	const Container &container_;
};

}
#endif
