//
// See the file LICENSE for redistribution information.
//
// Copyright (c) 2002-2005
//	Sleepycat Software.  All rights reserved.
//
// $Id: Statistics.cpp,v 1.52 2005/04/20 18:31:28 bostic Exp $
//

#include "dbxml_config.h"
#include "dbxml/XmlPortability.hpp"
#include "dbxml/XmlException.hpp"
#include "Statistics.hpp"
#include "SyntaxManager.hpp"
#include "Container.hpp"
#include "OperationContext.hpp"
#include "Cursor.hpp"
#include "Value.hpp"

#include <string>
#include <sstream>

using namespace std;
using namespace DbXml;

Statistics::Statistics(const Container &container, Transaction *txn,const Index &index,
		       const char *child, const char *parent, const XmlValue &value)
{
	StatisticsReadCache cache(container);
	OperationContext oc(txn);

	Key key;
	key.setIndex(index);

	if(!value.isNull() && AtomicTypeValue::convertToSyntaxType(value.getType()) != key.getSyntaxType()) {
		throw XmlException(XmlException::INVALID_VALUE, "Value type does not match index syntax type.");
	}
	if(!value.isNull() && key.getSyntaxType() == Syntax::NONE) {
		throw XmlException(XmlException::INVALID_VALUE, "A value has been specified for an index that does not require one.");
	}

	key.setIDsFromNames(oc, container, parent, child);
	if(index.getPath() == Index::PATH_EDGE && parent == 0) {
		key.setNodeLookup(true);
	}

	if(!value.isNull()) {
		key.setValue(value);
	}

	const KeyStatistics &stats = cache.getKeyStatistics(oc, key);

	numIndexedKeys_ = stats.numIndexedKeys_;
	numUniqueKeys_ = stats.numUniqueKeys_;
	sumKeyValueSize_ = stats.sumKeyValueSize_;

	if(index.getKey() == Index::KEY_EQUALITY && !value.isNull() &&
	   numUniqueKeys_ != 0.0) {
		numIndexedKeys_ /= numUniqueKeys_;
		sumKeyValueSize_ /= numUniqueKeys_;
		numUniqueKeys_ = 1.0;
	}
}

StatisticsWriteCache::StatisticsWriteCache()
	: dkv_(SyntaxManager::getInstance()->size())
{}

StatisticsWriteCache::~StatisticsWriteCache()
{
	reset();
}

void StatisticsWriteCache::reset()
{
	Dbt2KSMapVector::iterator i;
	for (i = dkv_.begin();i != dkv_.end();++i) {
		delete *i;
		*i = 0;
	}
}

// We assume that key has the correct endianness.
void StatisticsWriteCache::addToKeyStatistics(const Index &index, const Dbt &key, const Dbt &data, bool unique)
{
	const Syntax *syntax = SyntaxManager::getInstance()->getSyntax((Syntax::Type)index.getSyntax());

	Dbt2KSMap* dk = dkv_[syntax->getType()];
	if (dk == 0) {
		dk = new Dbt2KSMap;
		dkv_[syntax->getType()] = dk;
	}

	size_t skl = syntax->structureKeyLength(index, key.get_size());
	if (skl > 0 && skl <= key.get_size()) {
		DbtIn k(key.get_data(), skl);
		Dbt2KSMap::iterator i = dk->find(k);
		KeyStatistics *ps = 0;
		if (i == dk->end()) {
			KeyStatistics s;
			dk->insert(Dbt2KSMap::value_type(k, s));
			i = dk->find(k);
			ps = &i->second;
		} else {
			ps = &i->second;
		}
		if(index.indexerAdd()) {
			ps->numIndexedKeys_++;
			ps->sumKeyValueSize_ += key.get_size() + data.get_size();
			if (unique) {
				ps->numUniqueKeys_++;
			}
		} else {
			ps->numIndexedKeys_--;
			ps->sumKeyValueSize_ -= key.get_size() + data.get_size();
			if (unique) {
				ps->numUniqueKeys_--;
			}
		}
	}
}

int StatisticsWriteCache::updateContainer(OperationContext &context, Container &container) const
{
	int err = 0;
	int syntax = 0;
	Dbt2KSMapVector::const_iterator i;
	for (i = dkv_.begin();err == 0 && i != dkv_.end();++i) {
		Dbt2KSMap* dk = *i;
		if (dk != 0) {
			Dbt2KSMap::iterator i2;
			for (i2 = dk->begin();i2 != dk->end();++i2) {
				DbtIn &k = (DbtIn&)i2->first;  // jcm const_cast
				err = container.getIndexDB((Syntax::Type)syntax)->updateStatistics(context, k, i2->second);
				// JCM - err
			}
		}
		syntax++;
	}
	return err;
}

StatisticsReadCache::StatisticsReadCache(const Container &container)
	: container_(container)
{
}

void StatisticsReadCache::reset()
{
	statsMap_.clear();
}

const KeyStatistics &StatisticsReadCache::getKeyStatistics(OperationContext &context, const Key &key)
{
	Key noValueKey;
	if(key.getIndex().getKey() == Index::KEY_EQUALITY) {
		noValueKey.set(key, 0, 0);
	} else {
		noValueKey.set(key);
	}

	StatsMap::const_iterator it = statsMap_.find(noValueKey);
	if(it != statsMap_.end()) {
// 		cerr << "* hit: " << it->first.asString() << " -> " << it->second.asString() << endl;
		return it->second;
	} else {
		// Not found, so look it up
// 		cerr << "**** MISS: " << noValueKey.asString() << endl;
		populateStatistics(context, noValueKey);
// 		cerr << "    (found2: " << noValueKey.asString() << " -> " << statsMap_[noValueKey].asString() << ")\n";
		return statsMap_[noValueKey];
	}
}

double StatisticsReadCache::getPercentage(OperationContext &context, DbWrapper::Operation op1, Key k1, DbWrapper::Operation op2, Key k2)
{
	PercentageKey key(op1, k1, op2, k2);

	PercentageMap::const_iterator it = percentageMap_.find(key);
	if(it != percentageMap_.end()) {
// 		cerr << "    percentage hit: " << it->second << endl;
		return it->second;
	}

	const SyntaxDatabase *database = container_.getIndexDB(k1.getSyntaxType());
	double percentage;
	if(op2 == DbWrapper::NONE) {
		percentage = database->getStatisticsDB()->percentage(context, op1, DbWrapper::NONE, DbWrapper::NONE, k1, k2);
	} else {
		percentage = database->getStatisticsDB()->percentage(context, DbWrapper::RANGE, op1, op2, k1, k2);
	}
	putPercentage(key, percentage);
// 	cerr << "    percentage MISS: " << percentage << endl;
	return percentage;
}

void StatisticsReadCache::putKeyStatistics(const Key &key, const KeyStatistics &stats)
{
// 	cerr << "  inserting " << key.asString() << " -> " << stats.asString() << endl;
	statsMap_.insert(StatsMap::value_type(key, stats));
// 	cerr << "    (found: " << statsMap_[key].asString() << ")\n";
}

void StatisticsReadCache::putPercentage(const PercentageKey &pk, const double &p)
{
// 	cerr << "      inserting: " << p << endl;
	percentageMap_.insert(PercentageMap::value_type(pk, p));
}

void StatisticsReadCache::populateStatistics(OperationContext &context, const Key &key)
{
	SyntaxDatabase *database = const_cast<SyntaxDatabase*>(container_.getIndexDB(key.getSyntaxType()));

	if (!database)
		return;
	
	const Syntax *syntax = key.getSyntax();
	key.setDbtFromThis(context.key());

	int structureLength = syntax->structureKeyLength(key.getIndex(), context.key().get_size());
	if(structureLength < (int)context.key().get_size()) {
		context.key().set_size(structureLength); // trim the value off
	}

	Key tmpKey;
	KeyStatistics tmpStats;
	KeyStatistics result;

	// Behave like a prefix cursor, adding all the statistics with a key whose prefix matches our key
	int found = 0;
	Cursor cursor(database->getStatisticsDB()->getEnvironment(), database->getStatisticsDB()->getDb(), context.txn(), CURSOR_READ);
	int err = cursor.error();
	if(err == 0) {
		DbtOut original;
		original.set(context.key().get_data(), context.key().get_size());

		err = cursor.get(&context.key(), &context.data(), DB_SET_RANGE);
		while(err == 0) {
			if(context.key().get_size() < original.get_size() ||
			   memcmp(original.get_data(), context.key().get_data(), original.get_size()) != 0) {
				// We've reached the end, so set a flag saying so
				err = DB_NOTFOUND;
			}
			else {
				++found;

				// Cache the intermediate value (it might be useful)
				tmpKey.setThisFromDbt(context.key());
				tmpKey.getIndex().set(key.getSyntaxType(), Index::SYNTAX_MASK);
				tmpStats.setThisFromDbt(context.data());

				// Fix the unique keys value, if necessary
				if(tmpStats.numUniqueKeys_ == 0 &&
				   tmpStats.numIndexedKeys_ != 0) {
					tmpStats.numUniqueKeys_ = 1;
				}

				// Store the intermediate value
				putKeyStatistics(tmpKey, tmpStats);

				// add the value it to the results
				result.add(tmpStats);

				// Get the next key/data pair
				context.key().set(original.get_data(), original.get_size());
				err = cursor.get(&context.key(), &context.data(), DB_NEXT);
			}
		}
		if(err == DB_NOTFOUND || err == DB_KEYEMPTY) {
			err = 0;
		}
	}
	if(err) throw XmlException(err);

	if(found > 1) {
		putKeyStatistics(key, result);
	}
}

bool StatisticsReadCache::PercentageKey::operator<(const PercentageKey &o) const
{
	if(operation1 < o.operation1) return true;
	if(o.operation1 < operation1) return false;
	if(key1 < o.key1) return true;
	if(o.key1 < key1) return false;
	if(operation2 < o.operation2) return true;
	if(o.operation2 < operation2) return false;
	if(key2 < o.key2) return true;
	if(o.key2 < key2) return false;
	return false;
}

