//
// See the file LICENSE for redistribution information.
//
// Copyright (c) 2002-2005
//	Sleepycat Software.  All rights reserved.
//
// $Id: Results.cpp,v 1.60 2005/04/19 14:03:42 jsnelson Exp $
//

#include <xquery/XQQuery.hpp>

#include "dbxml_config.h"
#include "dbxml/XmlPortability.hpp"
#include "dbxml/XmlNamespace.hpp"
#include "dbxml/XmlException.hpp"
#include "Results.hpp"
#include "QueryContext.hpp"
#include "OperationContext.hpp"
#include "Container.hpp"
#include "QueryExecutionContext.hpp"
#include "QueryExpression.hpp"
#include "UTF8.hpp"
#include "Manager.hpp"
#include "HighResTimer.hpp"
#include "Document.hpp"

#include <xercesc/dom/DOM.hpp>
#if defined(XERCES_HAS_CPP_NAMESPACE)
  XERCES_CPP_NAMESPACE_USE
#endif

#include <pathan/PathanException.hpp>
#include <pathan/exceptions/DataItemException.hpp>

using namespace DbXml;
using namespace std;

// Results from Values
ValueResults::ValueResults()
	: vvi_(0)
{
}

ValueResults::~ValueResults()
{
	delete vvi_;
}

void ValueResults::reset()
{
	delete vvi_;
	vvi_ = 0;
}

size_t ValueResults::size() const
{
	return vv_.size();
}

int ValueResults::doNext(XmlValue &value, bool isPeek)
{
	if(vvi_ == 0) {
		if (vv_.size() == 0) {
			value = XmlValue();
			return 0;
		}
		vvi_ = new XmlValueVector::iterator;
		*vvi_ = vv_.begin();
	}

	if(*vvi_ != vv_.end()) {
		value = **vvi_;
		if (!isPeek)
			(*vvi_)++;
	} else {
		value = XmlValue();
	}

	return 0;
}

int ValueResults::next(XmlValue &value)
{
	return doNext(value, false);
}

int ValueResults::peek(XmlValue &value)
{
	return doNext(value, true);
}

bool ValueResults::hasNext()
{
	if(vvi_ == 0) {
		if (vv_.size() == 0)
			return false;
		vvi_ = new XmlValueVector::iterator;
		*vvi_ = vv_.begin();
	}
	if(*vvi_ != vv_.end())
		return true;
	return false;
}

int ValueResults::previous(XmlValue &value)
{
	if (hasPrevious()) { // share initialization code, below
		(*vvi_)--;
		value = **vvi_;
	} else {
		value = XmlValue();
	}
	return 0;
}

bool ValueResults::hasPrevious()
{
	if ((vvi_ == 0) ||
	    (*vvi_ == vv_.begin()))
		return false;
	return true;
}

void ValueResults::add(const XmlValue &value)
{
	vv_.push_back(value);
}

void ValueResults::add(const XmlValueVector &values)
{
	vv_.insert(vv_.end(), values.begin(), values.end());
}

// Results from DataItem tree, evaluated eagerly
EagerDIResults::EagerDIResults(QueryContext *context, Value *contextItem, QueryExpression &expr, Transaction *txn, u_int32_t flags)
{
	XmlQueryContext new_context = context->mergeIntoDynamicContext(expr.getContext());

	((QueryContext &)new_context).setTransaction(txn);

	ReferenceMinder evaluationMinder;
	((QueryContext &)new_context).setMinder(&evaluationMinder);
	((QueryContext &)new_context).setFlags(flags);

	((Manager &)((QueryContext &)new_context).getManager())
		.log(Log::C_QUERY, Log::L_INFO, "Starting eager query execution");

	HighResTimer t;
	t.start();

	try {
		XPath2MemoryManagerImpl mm;
		AutoRelease<XQContext> xpc(((QueryContext&)new_context).createDynamicXQContext(txn, expr.getXQContext(), mm));

		xpc->setContextItem(Value::convertToItem(contextItem, xpc));

		Result result(expr.getCompiledExpression()->evaluate(xpc));

		Item::Ptr item;
		while((item = result.next(xpc)) != NULLRCP) {
			if(item->isNode()) {
				const DOMNode *node = ((const Node*)(const Item*)item)->getDOMNode();
				Document *document = evaluationMinder.findDocument(node);
				if(document != 0) {
					vv_.push_back(Value::create(item, document, new_context));
				}
			}
			else {
				vv_.push_back(Value::create(item, xpc));
			}
		}
	}
	catch (const PathanException &e) {
		throw XmlException(XmlException::XPATH_EVALUATION_ERROR, XMLChToUTF8(e.getString()).str());
	}
	catch(const DSLException &e) {
		throw XmlException(XmlException::XPATH_EVALUATION_ERROR, XMLChToUTF8(e.getError()).str());
	}
	catch(XERCES_CPP_NAMESPACE_QUALIFIER DOMException &e) {
		throw XmlException(XmlException::XPATH_EVALUATION_ERROR, XMLChToUTF8(e.msg).str());
	}
	catch(XERCES_CPP_NAMESPACE_QUALIFIER XMLException &e) {
		throw XmlException(XmlException::XPATH_EVALUATION_ERROR, XMLChToUTF8(e.getMessage()).str());
	}

	t.stop();

	if(Log::isLogEnabled(Log::C_QUERY, Log::L_INFO)) {
		ostringstream s;
		s << "Finished eager query execution, time taken = " << (t.durationInSeconds() * 1000) << "ms";
		((Manager &)((QueryContext &)new_context).getManager())
			.log(Log::C_QUERY, Log::L_INFO, s);
	}
}

// Lazily evaluated results
void LazyResults::add(const XmlValue &value)
{
	throw XmlException(XmlException::LAZY_EVALUATION, "This result set is lazily evaluated. add() can only be called for eagerly evaluated result sets.");
}

void LazyResults::add(const XmlValueVector &vv)
{
	throw XmlException(XmlException::LAZY_EVALUATION, "This result set is lazily evaluated. add() can only be called for eagerly evaluated result sets.");
}

size_t LazyResults::size() const
{
	throw XmlException(XmlException::LAZY_EVALUATION, "This result set is lazily evaluated. size() can only be called for eagerly evaluated result sets.");
	return 0;
}

// Results from DataItem tree, evaluated lazily
LazyDIResults::LazyDIResults(QueryContext *context, Value *contextItem, QueryExpression &expr, Transaction *txn, u_int32_t flags)
	: context_(context->mergeIntoDynamicContext(expr.getContext())),
	  expr_(&expr),
	  contextItem_(contextItem == 0 ? Value::create() : contextItem),
	  xqc_(0),
	  result_(0),
	  nextItem_(0)
{
	((QueryContext &)context_).setTransaction(txn);
	((QueryContext &)context_).setMinder(&evaluationMinder_);
	((QueryContext &)context_).setFlags(flags);

	reset();
}

LazyDIResults::~LazyDIResults()
{
	result_ = 0; // Destruct the Result tree before the context
	if(xqc_ != 0) xqc_->release();
}

int LazyDIResults::next(XmlValue &value)
{
	timer_.start();

	try {
		Item::Ptr item;
		if (nextItem_ != NULLRCP) {
			item = nextItem_;
			nextItem_ = 0;
		} else
			item = result_.next(xqc_);
		if(item == NULLRCP) {
			value = Value::create();
		} else {
			if(item->isNode()) {
				const DOMNode *node = ((const Node*)(const Item*)item)->getDOMNode();
				Document *document = evaluationMinder_.findDocument(node);
				if(document != 0) {
					value = Value::create(item, document, context_);
				}
			}
			else {
				value = Value::create(item, xqc_);
			}
		}
	}
	catch (const PathanException &e) {
		throw XmlException(XmlException::XPATH_EVALUATION_ERROR, XMLChToUTF8(e.getString()).str());
	}
	catch(const DSLException &e) {
		throw XmlException(XmlException::XPATH_EVALUATION_ERROR, XMLChToUTF8(e.getError()).str());
	}
	catch(XERCES_CPP_NAMESPACE_QUALIFIER DOMException &e) {
		throw XmlException(XmlException::XPATH_EVALUATION_ERROR, XMLChToUTF8(e.msg).str());
	}
	catch(XERCES_CPP_NAMESPACE_QUALIFIER XMLException &e) {
		throw XmlException(XmlException::XPATH_EVALUATION_ERROR, XMLChToUTF8(e.getMessage()).str());
	}

	timer_.stop();

	if(value.isNull() && !result_.isNull()) {
		result_ = 0;
		if(Log::isLogEnabled(Log::C_QUERY, Log::L_INFO)) {
			ostringstream s;
			s << "Finished lazy query execution, time taken = "
			  << (timer_.durationInSeconds() * 1000) << "ms";
			((Manager &)((QueryContext &)context_).getManager())
				.log(Log::C_QUERY, Log::L_INFO, s);
		}
	}

	return 0;
}

static void
_throwNotImpl(const char * op)
{
	ostringstream s;
	s << "Operation not supported on Lazy XmlResults: ";
	s << op;
	throw XmlException(XmlException::INVALID_VALUE, s.str().c_str());
}

int LazyDIResults::peek(XmlValue &value)
{
	Item::Ptr item = nextItem_;
	if (!item) {
		nextItem_ = result_.next(xqc_);
		item = nextItem_;
	}
	int ret = next(value);
	// need to reset nextItem, so iterator doesn't really move
	nextItem_ = item;
	return ret;
}

int LazyDIResults::previous(XmlValue &)
{
	_throwNotImpl("previous");
	return 0;
}

bool LazyDIResults::hasNext()
{
	if (nextItem_ == NULLRCP)
		nextItem_ = result_.next(xqc_);
	if (nextItem_ == NULLRCP)
		return false;
	return true;
}

bool LazyDIResults::hasPrevious()
{
	_throwNotImpl("hasPrevious");
	return false;
}
	
	
void LazyDIResults::reset()
{
	((Manager &)((QueryContext &)context_).getManager())
		.log(Log::C_QUERY, Log::L_INFO, "Starting lazy query execution");

	timer_.reset();
	timer_.start();

	try {
		if(xqc_ != 0) xqc_->release();
		xqc_ = ((QueryContext&)context_).createDynamicXQContext(((QueryContext &)context_).getOperationContext().txn(),
									((QueryExpression*)expr_)->getXQContext(), mm_);
		xqc_->setContextItem(Value::convertToItem((Value*)contextItem_, xqc_));

		result_ = ((QueryExpression*)expr_)->getCompiledExpression()->evaluate(xqc_);
	}
	catch (const PathanException &e) {
		throw XmlException(XmlException::XPATH_EVALUATION_ERROR, XMLChToUTF8(e.getString()).str());
	}
	catch(const DSLException &e) {
		throw XmlException(XmlException::XPATH_EVALUATION_ERROR, XMLChToUTF8(e.getError()).str());
	}
	catch(XERCES_CPP_NAMESPACE_QUALIFIER DOMException &e) {
		throw XmlException(XmlException::XPATH_EVALUATION_ERROR, XMLChToUTF8(e.msg).str());
	}
	catch(XERCES_CPP_NAMESPACE_QUALIFIER XMLException &e) {
		throw XmlException(XmlException::XPATH_EVALUATION_ERROR, XMLChToUTF8(e.getMessage()).str());
	}

	timer_.stop();
}

// Results from a lazily manifested IDS
LazyIndexResults::LazyIndexResults(Container &container,
				   QueryContext *context,
				   Transaction *txn, const Index &index,
				   const Name &child, const char *parent,
				   const XmlValue &value, u_int32_t flags)
	: context_(context->copy()),
	  container_(&((TransactedContainer &)container)),
	  index_(index),
	  child_(child),
	  data_(new IndexData)
{
	((Manager &)((QueryContext &)context_).getManager())
		.log(Log::C_QUERY, Log::L_INFO, "Starting lazy index lookup");

	((QueryContext &)context_).setTransaction(txn);
	((QueryContext &)context_).setFlags(flags);
	OperationContext &oc = ((QueryContext*)context_)->getOperationContext();

	timer_.start();

	Key key;
	DbWrapper::Operation op = DbWrapper::EQUALITY;

	key.setIndex(index);

	if(!value.isNull() && AtomicTypeValue::convertToSyntaxType(value.getType()) != key.getSyntaxType()) {
		throw XmlException(XmlException::INVALID_VALUE, "Value type does not match index syntax type.");
	}
	if(!value.isNull() && key.getSyntaxType() == Syntax::NONE) {
		throw XmlException(XmlException::INVALID_VALUE, "A value has been specified for an index that does not require one.");
	}

	key.setIDsFromNames(oc, container_, parent, child.getURIName().c_str());
	if(index.getPath() == Index::PATH_EDGE && parent == 0) {
		key.setNodeLookup(true);
	}

	if(!value.isNull()) {
		key.setValue(value);
	}
	else if(index.getKey() != Index::KEY_PRESENCE) {
		op = DbWrapper::PREFIX;
	}

	int err = ((Container*)container_)->getIndexDB(key.getSyntaxType())->getIndexData(oc, data_, op, key);
	if(err) throw XmlException(err);

	timer_.stop();

	it_ = data_->begin();
}

int LazyIndexResults::doNext(XmlValue &value, bool isPeek)
{
	timer_.start();

	if(it_ != data_->end()) {
		OperationContext &oc = ((QueryContext*)context_)->
			getOperationContext();
		XmlDocument document = ((QueryContext&)context_).getManager().
			createDocument();
		int err = ((Container*)container_)->
			getDocument(oc, (*it_)->getDocID(), document,
				    ((QueryContext*)context_)->getFlags());
		if(err != 0) {
			((Manager &)((QueryContext &)context_).getManager())
				.log(Log::C_QUERY, Log::L_ERROR,
				     "Invalid index values found during query (document not found)");
			throw XmlException(XmlException::INTERNAL_ERROR,
					   "Invalid indexes");
		}

		if((*it_)->isSpecified(IndexEntry::NODE_ID)) {
			DOMElement *element = ((Document&)document).
				getElement(nsGetNid((*it_)->getNode()));

			if(element == 0) {
				((Manager &)((QueryContext &)context_).
				 getManager())
					.log(Log::C_QUERY, Log::L_ERROR,
					     "Invalid index values found during query (element not found)");
				throw XmlException(XmlException::INTERNAL_ERROR,
						   "Invalid indexes");
			}

			if(index_.getNode() == Index::NODE_ELEMENT) {
				value = new NodeValue(element, document);
			} else {
				DOMAttr *attr = element->
					getAttributeNodeNS(
						UTF8ToXMLCh(child_.getURI()).
						str(),
						UTF8ToXMLCh(child_.
							    getName()).str());
				if(attr == 0) {
					((Manager &)((QueryContext &)context_).getManager())
						.log(Log::C_QUERY, Log::L_ERROR,
						     "Invalid index values found during query (attribute not found)");
					throw XmlException(XmlException::INTERNAL_ERROR,
							   "Invalid indexes");
				}

				value = new NodeValue(attr, document);
			}
		}
		else {
			value = new NodeValue(document);
		}

		if (!isPeek)
			++it_;
	} else {
		value = Value::create();
		if(Log::isLogEnabled(Log::C_QUERY, Log::L_INFO)) {
			ostringstream s;
			s << "Finished lazy index lookup, time taken = "
			  << (timer_.durationInSeconds() * 1000) << "ms";
			((Manager &)((QueryContext &)context_).getManager())
				.log(Log::C_QUERY, Log::L_INFO, s);
		}
	}

	timer_.stop();

	return 0;
}

int LazyIndexResults::next(XmlValue &value)
{
	return doNext(value, false);
}

int LazyIndexResults::previous(XmlValue &value)
{
	--it_;
	return doNext(value, true); // set peek
}

int LazyIndexResults::peek(XmlValue &value)
{
	return doNext(value, true);
}

bool LazyIndexResults::hasNext()
{
	if (it_ != data_->end())
		return true;
	return false;
}

bool LazyIndexResults::hasPrevious()
{
	if (it_ != data_->begin())
		return true;
	return false;
}

void LazyIndexResults::reset()
{
	timer_.reset();
	it_ = data_->begin();
}

// Results from a eagerly manifested IDS
EagerIndexResults::EagerIndexResults(const Container &container,
				     QueryContext *context,
				     Transaction *txn, const Index &index,
				     const Name &child, const char *parent,
				     const XmlValue &value, u_int32_t flags)
{
	XmlQueryContext new_context = context->copy();

	((Manager &)((QueryContext &)new_context).getManager())
		.log(Log::C_QUERY, Log::L_INFO, "Starting eager index lookup");

	HighResTimer t;
	t.start();

	Key key;
	DbWrapper::Operation op = DbWrapper::EQUALITY;

	key.setIndex(index);

	if(!value.isNull() &&
	   AtomicTypeValue::convertToSyntaxType(value.getType()) !=
	   key.getSyntaxType()) {
		throw XmlException(XmlException::INVALID_VALUE, "Value type does not match index syntax type.");
	}
	if(!value.isNull() && key.getSyntaxType() == Syntax::NONE) {
		throw XmlException(XmlException::INVALID_VALUE, "A value has been specified for an index that does not require one.");
	}

	OperationContext &oc =
		((QueryContext*)new_context)->getOperationContext();
	oc.set(txn);

	key.setIDsFromNames(oc, container, parent, child.getURIName().c_str());
	if(index.getPath() == Index::PATH_EDGE && parent == 0) {
		key.setNodeLookup(true);
		op = DbWrapper::PREFIX;
	}

	if(!value.isNull()) {
		key.setValue(value);
	} else if(index.getKey() != Index::KEY_PRESENCE) {
		op = DbWrapper::PREFIX;
	}

	IndexData::SharedPtr data(new IndexData);
	const SyntaxDatabase *sdb = container.getIndexDB(key.getSyntaxType());
	if (!sdb)
		return; // index doesn't exist
	int err = sdb->getIndexData(oc, data, op, key);
	if(err)
		throw XmlException(err);

	IndexData::iterator end = data->end();
	for(IndexData::iterator it = data->begin(); it != end; ++it) {
		XmlDocument document = ((QueryContext&)new_context).getManager().createDocument();
		int err = container.getDocument(oc, (*it)->getDocID(),
						document, flags);
		if(err != 0) {
			((Manager &)((QueryContext &)new_context).getManager())
				.log(Log::C_QUERY, Log::L_ERROR, "Invalid index values found during query (document not found)");
			throw XmlException(XmlException::INTERNAL_ERROR, "Invalid indexes");
		}

		if((*it)->isSpecified(IndexEntry::NODE_ID)) {
			DOMElement *element = ((Document&)document).
				getElement(nsGetNid((*it)->getNode()));

			if(element == 0) {
				((Manager &)((QueryContext &)new_context).getManager())
					.log(Log::C_QUERY, Log::L_ERROR, "Invalid index values found during query (element not found)");
				throw XmlException(XmlException::INTERNAL_ERROR, "Invalid indexes");
			}

			if(index.getNode() == Index::NODE_ELEMENT) {
				vv_.push_back(new NodeValue(element, document));
			}
			else {
				DOMAttr *attr = element->
					getAttributeNodeNS(*child.getURI() == 0 ? 0 :
							   UTF8ToXMLCh(child.getURI()).str(),
							   UTF8ToXMLCh(child.getName()).str());
				if(attr == 0) {
					((Manager &)((QueryContext &)new_context).getManager())
						.log(Log::C_QUERY, Log::L_ERROR, "Invalid index values found during query (attribute not found)");
					throw XmlException(XmlException::INTERNAL_ERROR, "Invalid indexes");
				}

				vv_.push_back(new NodeValue(attr, document));
			}
		}
		else {
			vv_.push_back(new NodeValue(document));
		}
	}

	t.stop();

	if(Log::isLogEnabled(Log::C_QUERY, Log::L_INFO)) {
		ostringstream s;
		s << "Finished eager index lookup, time taken = " << (t.durationInSeconds() * 1000) << "ms";
		((Manager &)((QueryContext &)new_context).getManager())
			.log(Log::C_QUERY, Log::L_INFO, s);
	}
}
