//
// See the file LICENSE for redistribution information.
//
// Copyright (c) 2002-2005
//	Sleepycat Software.  All rights reserved.
//
// $Id: QueryExpression.cpp,v 1.35 2005/04/20 18:31:28 bostic Exp $
//

#include <assert.h>
#include <sstream>

#include <xquery/XQContext.hpp>
#include <xquery/XQEvaluator.hpp>
#include <xquery/XQQuery.hpp>
#include <xquery/XQException.hpp>

#include "dbxml_config.h"
#include "dbxml/XmlPortability.hpp"
#include "QueryExpression.hpp"
#include "QueryContext.hpp"
#include "Results.hpp"
#include "optimizer/CollectionAndDocOptimizer.hpp"
#include "optimizer/ContainerContextOptimizer.hpp"
#include "optimizer/QueryPlanGenerator.hpp"
#include "PrintDataItemTree.hpp"
#include "Manager.hpp"
#include "HighResTimer.hpp"

#include <pathan/internal/XPath2MemoryManagerImpl.hpp>
#include <pathan/DynamicContext.hpp>
#include <pathan/PathanEngine.hpp>
#include <pathan/PathanException.hpp>
#include <pathan/exceptions/DSLException.hpp>

#include <xquery/XQContext.hpp>

#include "UTF8.hpp"

using namespace DbXml;
using namespace std;

// QueryExpression
QueryExpression::QueryExpression(const std::string &query, XmlQueryContext &context)
	: query_(query),
	  context_(((QueryContext &)context).copy()),
	  memMgr_(new XPath2MemoryManagerImpl),
	  xqContext_(0),
	  expr_(0)
{
}

QueryExpression::~QueryExpression()
{
	xqContext_->release();

	// Deleting the memory manager also deletes
	// anything created with it.
	delete memMgr_;
}

std::string QueryExpression::getQueryPlan() const
{
	return DbXmlPrintDataItemTree::print(getCompiledExpression(), getXQContext());
}

XQContext *QueryExpression::getXQContext() const
{
	return xqContext_;
}

XPath2MemoryManager *QueryExpression::getMemMgr()
{
	return memMgr_;
}

void QueryExpression::init(Transaction *txn)
{
	xqContext_ = ((QueryContext &)getContext()).createXQContext(txn, *getMemMgr());

	((Manager &)((QueryContext &)getContext()).getManager())
		.log(Log::C_OPTIMIZER, Log::L_INFO, "Started parse");

	HighResTimer t;
	t.start();

	((QueryContext &)getContext()).setMinder(&minder_);

	try {
		scoped_ptr<Optimizer> optimizer(createOptimizer(txn));
		expr_ = XQEvaluator::parse(UTF8ToXMLCh(getQuery()).str(), getXQContext(), false);
		optimizer->startOptimize(expr_);
	}
	catch(const XQException &e) {
		ostringstream s;
		XMLChToUTF8 reason(e.getError());
		s << "Error in XQuery expression: " <<
			reason.str() << ", line " <<
			e.m_nLine << ", column " << e.m_nColumn;
		throw XmlException(XmlException::XPATH_PARSER_ERROR,
				   s.str());
	}
	t.stop();

	if(Log::isLogEnabled(Log::C_OPTIMIZER, Log::L_INFO)) {
		ostringstream s;
		s << "Finished parse, time taken = " << (t.durationInSeconds() * 1000) << "ms";
		((Manager &)((QueryContext &)getContext()).getManager())
			.log(Log::C_OPTIMIZER, Log::L_INFO, s);
	}
}

XQQuery *QueryExpression::getCompiledExpression() const
{
	assert(expr_ != 0);
	return expr_;
}

Results *QueryExpression::execute(Transaction *txn, Value *contextItem, XmlQueryContext &context, u_int32_t flags)
{
	if(context.getEvaluationType() == XmlQueryContext::Eager) {
		return new EagerDIResults(context, contextItem, *this, txn, flags);
	}
	else {
		return new LazyDIResults(context, contextItem, *this, txn, flags);
	}
}

DatabaseQueryExpression::DatabaseQueryExpression(const std::string &query, XmlQueryContext &context, Transaction *txn)
	: QueryExpression(query, context)
{
	init(txn);
}

Optimizer *DatabaseQueryExpression::createOptimizer(Transaction *txn)
{
	//   OutputTree *before = new OutputTree("Initial tree", getXQContext());
	//   StaticResolver *sr = new StaticResolver(getXQContext(), before);
	//   OutputTree *afterStatic = new OutputTree("After Static Resolution", getXQContext(), sr);
	//   CollectionAndDocOptimizer *cad = new CollectionAndDocOptimizer(minder_, getXQContext(), afterStatic);
	//   OutputTree *afterCandD = new OutputTree("After Collection And Doc Optimisation", getXQContext(), cad);
	//   QueryPlanGenerator *qpGen = new QueryPlanGenerator(0, getContext(), getXQContext(), afterCandD);
	//   return new OutputTree("After Raw Query Plan Generation", getXQContext(), qpGen);

	StaticResolver *sr = new StaticResolver(getXQContext());
	CollectionAndDocOptimizer *cdo = new CollectionAndDocOptimizer(minder_, getXQContext(), sr);
	QueryPlanGenerator *qpg = new QueryPlanGenerator(((QueryContext&)getContext()).getManager(), txn, getXQContext(), cdo);
	return qpg;
}

ContainerQueryExpression::ContainerQueryExpression(const std::string &containerName, const std::string &query, XmlQueryContext &context, Transaction *txn)
	: QueryExpression(query, context),
	  containerName_(containerName)
{
	init(txn);
}

Optimizer *ContainerQueryExpression::createOptimizer(Transaction *txn)
{
	//   OutputTree *before = new OutputTree("Initial tree", getXQContext());
	//   ContainerContextOptimizer *containerise = new ContainerContextOptimizer(containerName_, documentMode, getXQContext(), before);
	//   OutputTree *afterContainerise = new OutputTree("After Containerisation", getXQContext(), containerise);
	//   StaticResolver *sr = new StaticResolver(getXQContext(), afterContainerise);
	//   OutputTree *afterStatic = new OutputTree("After Static Resolution", getXQContext(), sr);
	//   CollectionAndDocOptimizer *cdo = new CollectionAndDocOptimizer(minder_, txn, getContext(), getXQContext(), afterStatic);
	//   OutputTree *afterCdo = new OutputTree("After Collection And Doc Optimizer", getXQContext(), sr);
	//   RawQueryPlanGenerator *rqpg = new RawQueryPlanGenerator(txn, getContext(), getXQContext(), afterCdo);
	//   return new OutputTree("After Query Plan Generation", getXQContext(), rqpg);

	ContainerContextOptimizer *cco = new ContainerContextOptimizer(containerName_, false, getXQContext());
	StaticResolver *sr = new StaticResolver(getXQContext(), cco);
	CollectionAndDocOptimizer *cdo = new CollectionAndDocOptimizer(minder_, getXQContext(), sr);
	QueryPlanGenerator *qpg = new QueryPlanGenerator(((QueryContext&)getContext()).getManager(), txn, getXQContext(), cdo);
	return qpg;
}
