//
// See the file LICENSE for redistribution information.
//
// Copyright (c) 2002-2005
//	Sleepycat Software.  All rights reserved.
//
// $Id: PrintDataItemTree.cpp,v 1.15 2005/04/14 22:08:53 gmf Exp $
//

#include <iostream>
#include <sstream>

#include "PrintDataItemTree.hpp"
#include "UTF8.hpp"
#include "dataItem/QueryPlanFunction.hpp"
#include "QueryPlan.hpp"

#include <xquery/XQQuery.hpp>
#include <xquery/XQContext.hpp>
#include <xquery/dataItem/XQGlobalVariable.hpp>

#include <pathan/dataItem/DataItem.hpp>
#include <pathan/dataItem/DataItemFunction.hpp>
#include <pathan/DynamicContext.hpp>

using namespace DbXml;
using namespace std;

static const int INDENT = 1;

string DbXmlPrintDataItemTree::print(const PathanExpression *expr, const DynamicContext *context, int indent)
{
	return "Cannot print PathanExpression in BDB XML";
}

string DbXmlPrintDataItemTree::print(const XQQuery *query, const DynamicContext *context, int indent)
{
	ostringstream s;

	s << getIndent(indent) << "<XQuery";
	if(query->getIsLibraryModule()) {
		s << " module=\"true\"";
	}
	if(query->getModuleTargetNamespace()) {
		s << " targetNamespace=\"" << XMLChToUTF8(query->getModuleTargetNamespace()).str() << "\"";
	}
	s << ">" << endl;
	indent += INDENT;

	DbXmlPrintDataItemTree p;

	for(std::vector<XQFunction*, PathanAllocator<XQFunction*> >::const_iterator i = query->m_userDefFns.begin();
	    i != query->m_userDefFns.end(); ++i) {
		XQFunction *f = *i;

		const XMLCh *funUri = f->getURI();
		const XMLCh *funName = f->getName();

		string name("{");
		name += XMLChToUTF8(funUri).str();
		name += "}:";
		name += XMLChToUTF8(funName).str();

		s << "  <FunctionDefinition name=\"" << name << "\">" << endl;
		s << p.printDataItem(f->getFunctionBody(), context, 2);
		s << "  </FunctionDefinition>" << endl;
	  }
	for(vector<XQGlobalVariable*, PathanAllocator<XQGlobalVariable*> >::const_iterator it = query->m_userDefVars.begin();
	    it != query->m_userDefVars.end(); ++it) {
		s << p.printDataItem(*it, context, indent);
	}
	s << p.printDataItem(query->getQueryBody(), context, indent);

	indent -= INDENT;
	s << getIndent(indent) << "</XQuery>" << endl;

	return s.str();
}

string DbXmlPrintDataItemTree::print(const DataItem *item, const DynamicContext *context, int indent)
{
	DbXmlPrintDataItemTree p;
	return p.printDataItem(item, context, indent);
}

string DbXmlPrintDataItemTree::printDataItem(const DataItem *item, const DynamicContext *context, int indent)
{
	switch(item->getType()) {
	case DataItem::LITERAL: {
		return printLiteral((DataItemLiteral *)item, context, indent);
		break;
	}
	case DataItem::SEQUENCE: {
		return printSequence((DataItemSequence *)item, context, indent);
		break;
	}
	case DataItem::FUNCTION: {
		return printFunction((DataItemFunction *)item, context, indent);
		break;
	}
	case DataItem::NAVIGATION: {
		return printNav((DataItemNav *)item, context, indent);
		break;
	}
	case DataItem::VARIABLE: {
		return printVariable((DataItemVariable *)item, context, indent);
		break;
	}
	case DataItem::STEP: {
		return printStep((DataItemStep *)item, context, indent);
		break;
	}
	case DataItem::IF: {
		return printIf((DataItemIf *)item, context, indent);
		break;
	}
	case DataItem::INSTANCE_OF: {
		return printInstanceOf((DataItemInstanceOf *)item, context, indent);
		break;
	}
	case DataItem::CASTABLE_AS: {
		return printCastableAs((DataItemCastableAs *)item, context, indent);
		break;
	}
	case DataItem::CAST_AS: {
		return printCastAs((DataItemCastAs *)item, context, indent);
		break;
	}
	case DataItem::TREAT_AS: {
		return printTreatAs((DataItemTreatAs *)item, context, indent);
		break;
	}
	case DataItem::PARENTHESIZED: {
		return printParenthesized((DataItemParenthesizedExpr *)item, context, indent);
		break;
	}
	case DataItem::FOR: {
		return printFor((DataItemFor *)item, context, indent);
		break;
	}
	case DataItem::QUANTIFIED: {
		return printQuantified((DataItemQuantifiedExpr *)item, context, indent);
		break;
	}
	case DataItem::OPERATOR: {
		return printOperator((DataItemOperator *)item, context, indent);
		break;
	}
	case DataItem::CONTEXT_ITEM: {
		return printContextItem((DataItemContextItem *)item, context, indent);
		break;
	}
	case ((DataItem::whichType)XQContext::DEBUG_HOOK): {
		return printDebugHook((XQDebugHook *)item, context, indent);
		break;
	}
	case ((DataItem::whichType)XQContext::FLWOR): {
		return printFLWOR((XQFLWOR *)item, context, indent);
		break;
	}
	case ((DataItem::whichType)XQContext::FLWOR_QUANTIFIED): {
		return printFLWORQuantified((XQQuantified *)item, context, indent);
		break;
	}
	case ((DataItem::whichType)XQContext::TYPESWITCH): {
		return printTypeswitch((XQTypeswitch *)item, context, indent);
		break;
	}
	case ((DataItem::whichType)XQContext::VALIDATE): {
		return printValidate((XQValidate *)item, context, indent);
		break;
	}
	case ((DataItem::whichType)XQContext::VARIABLE_DEFINITION): {
		return printGlobal((XQGlobalVariable *)item, context, indent);
		break;
	}
	case ((DataItem::whichType)XQContext::FUNCTION_CALL): {
		return printFunctionCall((XQFunctionCall *)item, context, indent);
		break;
	}
	case ((DataItem::whichType)XQContext::DOM_CONSTRUCTOR): {
		return printDOMConstructor((XQDOMConstructor *)item, context, indent);
		break;
	}
	case ((DataItem::whichType)XQContext::ORDERING_CHANGE): {
		return printOrderingChange((XQOrderingChange *)item, context, indent);
		break;
	}
	case ((DataItem::whichType)XQContext::USER_FUNCTION): {
		return printUserFunction((XQFunction::XQFunctionEvaluator *)item, context, indent);
		break;
	}
	}
	return getIndent(indent) + "<Unknown/>\n";
}

string DbXmlPrintDataItemTree::printFunction(const DataItemFunction *item, const DynamicContext *context, int indent)
{
	const XMLCh *funUri = item->getFunctionURI();
	const XMLCh *funName = item->getFunctionName();

	if(funUri == DataItemFunction::XMLChFunctionURI &&
	   funName == QueryPlanFunction::name) {
		return printQueryPlanFunction((QueryPlanFunction*)item, context, indent);
	}

	ostringstream s;

	string in(getIndent(indent));

	string name("{");
	name += XMLChToUTF8(funUri).str();
	name += "}:";
	name += XMLChToUTF8(funName).str();

	const VectorOfDataItems &args = item->getArguments();
	if(args.empty() && !hasPredicates(item)) {
		s << in << "<Function name=\"" << name << "\"/>" << endl;
	} else {
		s << in << "<Function name=\"" << name << "\">" << endl;
		for(VectorOfDataItems::const_iterator i = args.begin();
		    i != args.end(); ++i) {
			s << printDataItem(*i, context, indent + INDENT);
		}
		s << printPredicates(item, context, indent + INDENT);
		s << in << "</Function>" << endl;
	}

	return s.str();
}

string DbXmlPrintDataItemTree::printQueryPlanFunction(
	const QueryPlanFunction *item, const DynamicContext *context, int indent)
{
	ostringstream s;
	// For now, only print implied schema if debug; it's not being
	// used, and makes query plan output larger
#ifdef DEBUG
	static bool printImpliedSchema = true;
#else
	static bool printImpliedSchema = false;
#endif

	string in(getIndent(indent));

	string result(item->getFunctionType() == QueryPlanFunction::DOCUMENT ?
		      "document" : "collection");

	s << in << "<QueryPlanFunction result=\"" << result << "\">" << endl;

	string planString = item->isQueryPlanExecutable() ? "OQPlan" : "RQPlan";
	if(item->getQueryPlan() != 0) {
		s << in << "  <" << planString << ">" <<
			item->getQueryPlan()->toString(false) << "</" <<
			planString << ">" << endl;
	} else {
		s << in << "  <" << planString << "/>" << endl;
	}
	// For now, don't print implied schema; it's not being
	// used, and makes query plan output larger
	if (printImpliedSchema) {
		if(item->getImpliedSchema() != 0) {
			s << in << "  <ImpliedSchema>" << endl;
			s << item->getImpliedSchema()->toString(
				indent + INDENT + INDENT);
			s << in << "  </ImpliedSchema>" << endl;
		} else {
			s << in << "  <ImpliedSchema/>" << endl;
		}
	}

	s << printPredicates(item, context, indent + INDENT);
	s << in << "</QueryPlanFunction>" << endl;

	return s.str();
}
