//
// See the file LICENSE for redistribution information.
//
// Copyright (c) 2002-2005
//	Sleepycat Software.  All rights reserved.
//
// $Id: Manager.cpp,v 1.16 2005/04/20 18:31:25 bostic Exp $
//

#include <dbxml/XmlPortability.hpp>
#include <assert.h>
#include "Manager.hpp"
#include "dbxml/XmlManager.hpp"
#include "dbxml/XmlException.hpp"
#include "dbxml/XmlException.hpp"
#include "Transaction.hpp"
#include "TransactedContainer.hpp"
#include "UTF8.hpp"
#include "Globals.hpp"
#include "SyntaxManager.hpp"
#include "dataItem/DbXmlURIResolver.hpp"

#if defined(XERCES_HAS_CPP_NAMESPACE)
XERCES_CPP_NAMESPACE_USE
#endif

using namespace DbXml;
using namespace std;

// Default environment cache size to 50MB
// This beats DB's default of 256K, and should not
// consume excessive resource
static const u_int32_t db_cachesize_default = 50 * 1024 * 1024;

Manager::Manager(DbEnv *dbEnv, u_int32_t flags)
	: dbEnv_(dbEnv),
	  dbEnvAdopted_((flags & DBXML_ADOPT_DBENV)!=0),
	  isTransacted_(false),
	  autoOpen_((flags & DBXML_ALLOW_AUTO_OPEN)!=0),
	  defaultContainerFlags_(0),
	  defaultPageSize_(0),
	  defaultContainerType_(XmlContainer::NodeContainer),
	  openContainers_()
{
	checkFlags(construct_manager_flag_info, "Manager()",
		   flags, DBXML_ADOPT_DBENV|
		   DBXML_ALLOW_EXTERNAL_ACCESS|DBXML_ALLOW_AUTO_OPEN);
	if(dbEnv_ == 0) {
		throw XmlException(
			XmlException::INVALID_VALUE,
			"Null DbEnv pointer passed as parameter to XmlManager.");
	}
	const char *dbhome = 0;
	dbEnv_->get_home(&dbhome);
	if (dbhome && *dbhome)
		dbHome_ = dbhome;
	u_int32_t dbflags = 0;
	dbEnv_->get_open_flags(&dbflags);
	if (dbflags & DB_INIT_TXN)
		isTransacted_ = true;
	if (dbflags & DB_AUTO_COMMIT) {
		throw XmlException(
			XmlException::INVALID_VALUE,
			"A DbEnv using DB_AUTO_COMMIT cannot be used to construct an XmlManager object.");
	}
	resolvers_ = new ResolverStore();
	resolvers_->setSecure((flags & DBXML_ALLOW_EXTERNAL_ACCESS) == 0);
	initialize(dbEnv_);
}

Manager::Manager(u_int32_t flags)
	: dbEnv_(new DbEnv(0)),
	  dbEnvAdopted_(true),
	  isTransacted_(false),
	  autoOpen_((flags & DBXML_ALLOW_AUTO_OPEN)!=0),
	  defaultContainerFlags_(0),
	  defaultPageSize_(0),
	  defaultContainerType_(XmlContainer::NodeContainer),
	  openContainers_()
{
	checkFlags(construct_manager_flag_info, "Manager()",
		   flags, DBXML_ALLOW_EXTERNAL_ACCESS|DBXML_ALLOW_AUTO_OPEN);
	resolvers_ = new ResolverStore();
	resolvers_->setSecure((flags & DBXML_ALLOW_EXTERNAL_ACCESS) == 0);
	initialize(dbEnv_); // initializes Xerces and Pathan
	// init DB defaults
	dbEnv_->set_cachesize(0, db_cachesize_default, 1);
	dbEnv_->set_errpfx("BDB XML");
	dbEnv_->set_error_stream(&cerr);
	try {
		dbEnv_->open(0, DB_PRIVATE|DB_CREATE|DB_INIT_MPOOL, 0);
	}
	catch(DbException &e) {
		terminate();
		throw XmlException(e);
	}
	catch(std::exception &e) {
		terminate();
		std::string error("Error opening default database environment");
		throw XmlException(
			XmlException::INTERNAL_ERROR, error,
			__FILE__, __LINE__);
	}
}

Manager::~Manager()
{
	openContainers_.releaseRegisteredContainers();

	try {
		if(dbEnvAdopted_) {
			dbEnv_->close(0);
			delete dbEnv_;
		}
	}
	catch(...) {
		// Destructors don't throw.
	}
	terminate();
	delete resolvers_;
}

void Manager::initialize(DbEnv *env)
{
	// Initialize the global variables
	Globals::initialize(env);
}

void Manager::terminate()
{
	// terminate the global variables
	Globals::terminate();
}

XERCES_CPP_NAMESPACE_QUALIFIER DatatypeValidatorFactory
&Manager::getDatatypeValidatorFactory() const
{
	return Globals::getDatatypeValidatorFactory();
}

TransactedContainer *Manager::openContainer(
	const std::string &name, Transaction *txn, u_int32_t flags,
	XmlContainer::ContainerType type,
	int mode, bool doVersionCheck)
{
	if(flags & ~(DB_XA_CREATE|DB_CREATE|
		     DB_DIRTY_READ|DB_EXCL|DB_NOMMAP|
		     DB_RDONLY|DB_THREAD|DBXML_CHKSUM|DBXML_ENCRYPT|
		     DBXML_INDEX_NODES|DBXML_ALLOW_VALIDATION|
		     DBXML_TRANSACTIONAL)) {
		throw XmlException(
			XmlException::INVALID_VALUE,
			"Invalid flags to method XmlManager::openContainer");
	}

	return openContainers_.findContainer(
		*this, name, txn, flags,
		type, defaultPageSize_, mode,
		doVersionCheck);
}

TransactedContainer *Manager::getOpenContainer(const std::string &name)
{
	return openContainers_.findOpenContainer(name);
}

void Manager::removeContainer(Transaction *txn, const std::string &name)
{
	int err = 0;
	try {
		err = dbEnv_->dbremove(Transaction::toDbTxn(txn), name.c_str(), 0, 0);
	}
	catch(DbException &e) {
		throw XmlException(e);
	}
	if(err) {
		throw XmlException(err);
	}
	else {
		ostringstream oss;
		oss << "Container '" << name << "' removed.";
		log(C_CONTAINER, L_DEBUG, oss);
	}
}

void Manager::renameContainer(Transaction *txn, const std::string &oldName,
			      const std::string &newName)
{
	int err = 0;
	try {
		err = dbEnv_->dbrename(Transaction::toDbTxn(txn),
				       oldName.c_str(), 0, newName.c_str(), 0);
	}
	catch(DbException &e) {
		throw XmlException(e);
	}
	if(err) {
		throw XmlException(err);
	}
	else {
		ostringstream oss;
		oss << "Container '" << oldName <<
			"' renamed to '" << newName << "'.";
		log(C_CONTAINER, L_DEBUG, oss);
	}
}

void Manager::upgradeContainer(const std::string &name,
				UpdateContext &context)
{
	int err = 0;
	//
	// Upgrade Berkeley DB
	//
	{
		Db db(dbEnv_, 0);
		err = db.upgrade(name.c_str(), 0);
	}
	//
	// Upgrade Berkeley DB XML
	//
	if(err == 0) {
		TransactedContainer *container =
			openContainer(name, 0, 0,
				      getDefaultContainerType(), 0, false);
		// Deletes the container at the end of this scope
		XmlContainer containerWrapper(container);

		unsigned int current_version, saved_version;
		container->getConfigurationDB()->
			getVersions(0, current_version, saved_version);
		// supported upgrades
		// version 3->4 (2.0->2.1)
		if (current_version == saved_version)
			return;
		if (current_version < saved_version) {
			ostringstream s;
			s << "Container version '";
			s << saved_version;
			s << "' is more recent than the bdbxml library version '";
			s << Container::version;
			s << "'.  Use a more recent release of the bdbxml library";
			throw XmlException(XmlException::VERSION_MISMATCH, s.str());
		}
		if (saved_version == VERSION_20 &&
		    current_version == VERSION_21) {
			container->upgrade(saved_version, current_version,
					   context);
		} else {
			throw XmlException(
				XmlException::VERSION_MISMATCH,
				"Upgrade is not supported from release 1.2.x to release 2.x.");
		}
	}
}

void Manager::checkFlags(const FlagInfo *flag_info, const char *function,
			 u_int32_t flags, u_int32_t mask) const
{
	Log::checkFlags(dbEnv_, C_MANAGER, 0, function,
			flag_info, flags, mask);
}

void Manager::log(ImplLogCategory c, ImplLogLevel l,
		   const ostringstream &s) const
{
	Log::log(dbEnv_, c, l, s.str().c_str());
}

void Manager::log(ImplLogCategory c, ImplLogLevel l, const string &s) const
{
	Log::log(dbEnv_, c, l, s.c_str());
}

Manager::ContainerStore::ContainerStore()
	: mutex_(XMLPlatformUtils::makeMutex())
{
}

Manager::ContainerStore::~ContainerStore()
{
	XMLPlatformUtils::closeMutex(mutex_);
}

int Manager::ContainerStore::closeContainer(TransactedContainer *container,
					     u_int32_t flags)
{
	int err = 0;
	try {
		MutexLock lock(mutex_);

		// remove name
		Map::iterator i = store_.find(container->getName());
		if(i != store_.end() && container == i->second) {
			store_.erase(i);
		}
#if 0
		// remove aliases
		if (container->hasAlias()) {
			for (i = store_.begin(); i != store_.end(); i++)
				if (i->second == container)
					store_.erase(i);
		}
#endif

	}
	catch (const XMLException &e) {
		throw XmlException(
			XmlException::INTERNAL_ERROR,
			string("Error during Xerces-C mutex actions: ") +
			XMLChToUTF8(e.getMessage()).str(), __FILE__, __LINE__);
	}
	return err;
}

bool Manager::ContainerStore::addAlias(const std::string &alias,
				       TransactedContainer *container)
{
	Map::iterator i = store_.find(alias);
	if(i == store_.end()) {
		store_[alias] = container;
		return true;
	}
	return false;
}

// only remove if the name matches the container
bool Manager::ContainerStore::removeAlias(const std::string &alias,
					  TransactedContainer *container)
{
	Map::iterator i = store_.find(alias);
	if(i != store_.end() && container == i->second) {
		store_.erase(i);
		return true;
	}
	return false;
}

TransactedContainer *Manager::ContainerStore::findContainer(
	Manager &db, const std::string &name, Transaction *txn,
	u_int32_t flags, XmlContainer::ContainerType type,
	u_int32_t pagesize, int mode, bool doVersionCheck)
{
	TransactedContainer *result = 0;

	try {
		MutexLock lock(mutex_);

		Map::iterator i = store_.find(name);
		if(i != store_.end()) {
			// Found one already open
			result = i->second;
		} else {
			// Have to open one ourselves...
			result = new TransactedContainer(
				db, name, txn, flags,
				pagesize, mode, type, doVersionCheck);
			store_[result->getName()] = result;
		}
	}
	catch(DbException &e) {
		throw XmlException(e);
	}
	catch (const XMLException &e) {
		throw XmlException(
			XmlException::INTERNAL_ERROR,
			string("Error during Xerces-C mutex actions: ") +
			XMLChToUTF8(e.getMessage()).str(), __FILE__, __LINE__);
	}

	return result;
}

TransactedContainer *Manager::ContainerStore::findOpenContainer(
	const std::string &name)
{
	TransactedContainer *result = NULL;
	MutexLock lock(mutex_);

	Map::iterator i = store_.find(name);
	if(i != store_.end()) {
		// Found one already open
		result = i->second;
	}
	return result;
}

void Manager::ContainerStore::releaseRegisteredContainers()
{
	try {
		MutexLock lock(mutex_);
		store_.clear();
  }
	catch (const XMLException &e) {
		throw XmlException(
			XmlException::INTERNAL_ERROR,
			string("Error during Xerces-C mutex actions: ") +
			XMLChToUTF8(e.getMessage()).str(), __FILE__, __LINE__);
	}
}
