//
// See the file LICENSE for redistribution information.
//
// Copyright (c) 2002-2005
//	Sleepycat Software.  All rights reserved.
//
// $Id: KeyStash.cpp,v 1.9 2005/04/05 16:44:01 bostic Exp $
//

#include "dbxml/XmlPortability.hpp"
#include "KeyStash.hpp"
#include "Log.hpp"
#include "Document.hpp"
#include "Container.hpp"
#include "IndexEntry.hpp"

#include <string>
#include <sstream>

using namespace DbXml;
using namespace std;

KeyStash::KeyStash()
	: buffer_(0, 1024)
{
}

KeyStash::~KeyStash()
{
	reset();
}

void KeyStash::reset()
{
	EntrySet::iterator end = keys_.end();
	for(EntrySet::iterator i = keys_.begin();
	    i != end; ++i) {
		delete *i;
	}
	keys_.clear();
	buffer_.reset();
}

void KeyStash::addEntry(Entry *toAdopt)
{
	pair<EntrySet::iterator, bool> inserted =
		keys_.insert(toAdopt);
	if(!inserted.second) {
		// the Entry is a duplicate, see if it cancels
		// out the existing entry
		if((*inserted.first)->index.indexerAdd() !=
		   toAdopt->index.indexerAdd()) {
			// They cancel each other out, so remove
			// the one from keys_
			delete *inserted.first;
			keys_.erase(inserted.first);
		}
		delete toAdopt;
	}
}

void KeyStash::addKey(const Key &key, const char *value,
		      size_t length, const IndexEntry &ie)
{
	addEntry(new Entry(key, value, length, ie, buffer_));
}

void KeyStash::addConsolationKey(const Key &key, const IndexEntry &ie)
{
	addEntry(new Entry(key, 0, 0, ie, buffer_));
}

int KeyStash::updateIndex(OperationContext &context, Container *container) const
{
	statistics_.reset();

	int err = 0;

	//
	// We iterate over the key stash, adding, or deleting the keys
	// from the indexes. The key buffer comes out of the key stash
	// with the correct endianness.
	//
	Index index;
	DbtIn key, data;
	EntrySet::const_iterator end = keys_.end();
	for(EntrySet::const_iterator i = keys_.begin();
	    i != end && err == 0; ++i) {
		// Add each key to the index.
		index = (*i)->index;
		(*i)->getKey(key);
		(*i)->getData(data);

		bool addKey = index.indexerAdd();
		if (Log::isLogEnabled(Log::C_INDEXER, Log::L_DEBUG)) {
			Buffer k(key.get_data(), key.get_size(),
				 /*wrapper*/true);
			Buffer d(data.get_data(), data.get_size(),
				 /*wrapper*/true);
			std::ostringstream oss;
			if (addKey)
				oss << "add " << index.asString()
				    << ", key={" << k.asStringBrief()
				    << "}, data={" << d.asStringBrief()
				    << "}";
			else
				oss << "delete " << index.asString()
				    << ", key={" << k.asStringBrief()
				    << "}, data={" << d.asStringBrief()
				    << "}";
			container->log(Log::C_INDEXER, Log::L_DEBUG, oss);
		}
		bool isEqualityKey =
			(index.getKey() == Index::KEY_EQUALITY);
		SyntaxDatabase *database = container->
			getIndexDB((Syntax::Type)index.getSyntax(),
				   context.txn(), true);
		bool duplicate = true;
		if (addKey) {
			if (isEqualityKey)
				duplicate = database->getIndexDB()->
					exists(context.txn(), key);
			// endianness of key buffer is correct
			err = database->getIndexDB()->
				putIndexEntry(context, key, data);
		} else {
			err = database->getIndexDB()->
				delIndexEntry(context, key, data);
			if (isEqualityKey)
				duplicate = database->getIndexDB()->
					exists(context.txn(), key);
		}
		if (err == 0) {
			statistics_.addToKeyStatistics(index, key, data,
						       !duplicate);
		}
	}
	//
	// Update the index key statistics stored in the container.
	//
	if (err == 0) {
		err = statistics_.updateContainer(context, *container);
	}

	return err;
}

KeyStash::Entry::Entry(const Key &key, const char *value,
		       size_t length, const IndexEntry &ie,
		       Buffer &b)
	: index(key.getIndex()),
	  key_offset(0),
	  key_length(0),
	  data_offset(0),
	  data_length(0),
	  buffer(b)
{
	key_offset = b.getCursorPosition();
	key_length = key.marshal(buffer, value, length);

	data_length = b.reserve(data_offset, ie.marshal(0, /*count*/true));
	ie.marshal((xmlbyte_t*)b.getBuffer(data_offset), /*count*/false);
}

bool KeyStash::EntryCompare::operator()(const Entry *a, const Entry *b) const
{
	if(a->key_length != b->key_length) return a->key_length < b->key_length;

	int key_cmp = ::memcmp(a->getKey(), b->getKey(), a->key_length);
	if(key_cmp != 0)
		return key_cmp < 0;

	if(a->data_length != b->data_length) return a->data_length < b->data_length;

	return ::memcmp(a->getData(), b->getData(), a->data_length) < 0;
}
