//
// See the file LICENSE for redistribution information.
//
// Copyright (c) 2002-2005
//	Sleepycat Software.  All rights reserved.
//
// $Id: IndexEntry.hpp,v 1.2 2005/04/05 16:44:01 bostic Exp $
//

#ifndef __INDEXENTRY_HPP
#define	__INDEXENTRY_HPP

#include <set>

#include "nodeStore/NsTypes.hpp"
#include "shared_ptr.hpp"

#include <xercesc/framework/MemoryManager.hpp>

class Dbt;

namespace DbXml
{

class DbtOut;

class IndexEntry : private XER_NS MemoryManager
{
public:
	/// Keep less than 1 byte
	enum Format {
		/// Has DOC_ID
		D_FORMAT = 0,
		/// Has DOC_ID, NODE_ID, LAST_CHILD_ID, and NODE_LEVEL
		DSEL_FORMAT = 1,

		KNOWN_FORMATS = 2
	};

	enum Info {
		NODE_ID = 0,
		LAST_CHILD_ID = 1,
		NODE_LEVEL = 2,
		PARENT_ID = 3,

		INFO_MAX = 4
	};

	IndexEntry();
	IndexEntry(Format format, uint32_t docid, const nsNode_t *node);
	~IndexEntry();

	void reset();

	Format getFormat() const { return format_; }
	uint32_t getDocID() const { return docid_; }
	const nsNode_t *getNode() const { return node_; }

	void set(uint32_t docid) { docid_ = docid; }
	void set(const nsNode_t *node);

	void setDbtFromThis(DbtOut &dbt) const;
	void setThisFromDbt(const Dbt &dbt);

	/// Returns the marshal size, if count is true
	int marshal(xmlbyte_t *buffer, bool count) const;

	bool isSpecified(Info info) const;
	static bool isSpecified(Format format, Info info);

	/// for sorting
	bool operator<(const IndexEntry &o) const;
private:
	IndexEntry(const IndexEntry &);
	IndexEntry &operator=(const IndexEntry &);

	void unmarshal(const Dbt &dbt);
	void allocateNsNode(xmlbyte_t **curP, xmlbyte_t **endP,
			    uint32_t dbtsize);
	void *allocate(size_t size);
	void deallocate(void *p);
private:
	Format format_;
	uint32_t docid_;
	nsNode_t *node_;
	bool owned_;

	/// A truth table for index entry formats
	static const bool indexFormats_[KNOWN_FORMATS][INFO_MAX];
};

struct IndexEntrySort
{
	bool operator()(const IndexEntry *a, const IndexEntry *b)
	{
		if(a == 0 || b == 0) return a < b;
		return *a < *b;
	}
};

class IndexData : public std::set<IndexEntry*, IndexEntrySort>
{
public:
	typedef shared_ptr<IndexData> SharedPtr;

	IndexData() {}
	~IndexData() {
		iterator e = end();
		for(iterator i = begin(); i != e; ++i)
			delete *i;
	}

private:
	IndexData(const IndexData &);
	IndexData &operator=(const IndexData &);
};

}

#endif
