//
// See the file LICENSE for redistribution information.
//
// Copyright (c) 2002-2005
//	Sleepycat Software.  All rights reserved.
//
// $Id: IndexEntry.cpp,v 1.4 2005/04/20 18:31:24 bostic Exp $
//

#include "nodeStore/NsUtil.hpp"
#include "IndexEntry.hpp"
#include "ScopedDbt.hpp"

using namespace DbXml;

IndexEntry::IndexEntry()
	: format_((Format)-1),
	  docid_(0),
	  node_(0),
	  owned_(false)
{
}

IndexEntry::IndexEntry(Format format, uint32_t docid, const nsNode_t *node)
	: format_(format),
	  docid_(docid),
	  node_((nsNode_t*)node),
	  owned_(false)
{
}

IndexEntry::~IndexEntry()
{
	reset();
}

void IndexEntry::reset()
{
	if(owned_) NsUtil::nsFreeNode(this, node_);
	format_ = (Format)-1;
	docid_ = 0;
	node_ = 0;
	owned_ = false;
}

void IndexEntry::set(const nsNode_t *node)
{
	if(owned_) NsUtil::nsFreeNode(this, node_);
	node_ = (nsNode_t*)node;
	owned_ = false;
}

void IndexEntry::setDbtFromThis(DbtOut &dbt) const
{
	int count = marshal(0, /*count*/true);
	dbt.set(0, count);
	marshal((xmlbyte_t*)dbt.get_data(), /*count*/false);
}

void IndexEntry::setThisFromDbt(const Dbt &dbt)
{
	reset();
	unmarshal(dbt);
}

int IndexEntry::marshal(xmlbyte_t *ptr, bool count) const
{
	int size = 0;

	if(count) {
		size += 1; // For the prefix byte
		size += NsUtil::nsCountInt(docid_);

		if(isSpecified(NODE_ID))
			size += NsUtil::nsCountId(nsGetNid(node_));
		if(isSpecified(LAST_CHILD_ID)) {
			const nid_t *lastchild = nsLastChildNid(node_);
			if(lastchild == 0) size += 1; // For a single null
			else size += NsUtil::nsCountId(lastchild);
		}
		if(isSpecified(NODE_LEVEL))
			size += NsUtil::nsCountInt(node_->nd_level);
		if(isSpecified(PARENT_ID))
			size += NsUtil::nsCountId(nsGetParentNid(node_));
	}
	else {
		*ptr++ = (xmlbyte_t)format_;
		ptr += NsUtil::nsMarshInt(ptr, docid_);

		if(isSpecified(NODE_ID))
			ptr += NsUtil::nsMarshId(ptr, nsGetNid(node_));
		if(isSpecified(LAST_CHILD_ID)) {
			const nid_t *lastchild = nsLastChildNid(node_);
			if(lastchild == 0) *ptr++ = 0; // Store a single null
			else ptr += NsUtil::nsMarshId(ptr, lastchild);
		}
		if(isSpecified(NODE_LEVEL))
			ptr += NsUtil::nsMarshInt(ptr, node_->nd_level);
		if(isSpecified(PARENT_ID))
			ptr += NsUtil::nsMarshId(ptr, nsGetParentNid(node_));
	}

	return size;
}

void IndexEntry::unmarshal(const Dbt &dbt)
{
	const xmlbyte_t *ptr = (const xmlbyte_t *)dbt.get_data();

	format_ = (Format)*ptr++;
	if(format_ >= KNOWN_FORMATS)
		NsUtil::nsThrowException(XmlException::INTERNAL_ERROR,
					 "unknown format in index entry",
					 __FILE__, __LINE__);
	ptr += NsUtil::nsUnmarshInt(ptr, &docid_);

	xmlbyte_t *curP = 0;
	xmlbyte_t *endP = 0;

	if(isSpecified(NODE_ID)) {
		allocateNsNode(&curP, &endP, dbt.get_size());
		ptr += NsUtil::nsUnmarshId(ptr, &node_->nd_header.nh_id,
					   &endP, /*copyStrings*/true);
	}
	if(isSpecified(LAST_CHILD_ID)) {
		if(*ptr == 0) {
			// There is no last child
			++ptr;
		}
		else {
			allocateNsNode(&curP, &endP, dbt.get_size());
			ptr += NsUtil::nsUnmarshId(ptr, &node_->nd_child->cl_child[0].ce_id,
						   &endP, /*copyStrings*/true);
		}
	}
	if(isSpecified(NODE_LEVEL)) {
		allocateNsNode(&curP, &endP, dbt.get_size());
		ptr += NsUtil::nsUnmarshInt(ptr, &node_->nd_level);
	}
	if(isSpecified(PARENT_ID)) {
		allocateNsNode(&curP, &endP, dbt.get_size());
		ptr += NsUtil::nsUnmarshId(ptr, &node_->nd_header.nh_parent,
					   &endP, /*copyStrings*/true);
	}
}

void IndexEntry::allocateNsNode(xmlbyte_t **curP, xmlbyte_t **endP, uint32_t dbtsize)
{
	if(node_ == 0) {
		// The nsNode_t is allocated in one large chunk, with enough
		// room for the nsNode_t, an nsChildList_t and the strings
		// it is going to need.
		uint32_t allocSize = dbtsize + sizeof(nsNode_t) +
			sizeof(nsChildList_t);
		*curP = (xmlbyte_t *)allocate(allocSize);
		if (!*curP)
			NsUtil::nsThrowException(XmlException::NO_MEMORY_ERROR,
						 "allocateNsNode failed to allocate memory",
						 __FILE__, __LINE__);
		memset(*curP, 0, allocSize);
		*endP = *curP + allocSize - 1;

		node_ = (nsNode_t*)*curP;
		*curP = (xmlbyte_t*)(node_ + 1);

		node_->nd_child = (nsChildList_t*)*curP;
		*curP = (xmlbyte_t*)(node_->nd_child + 1);
		node_->nd_child->cl_numChild = 1;
		node_->nd_child->cl_maxChild = 1;

		node_->nd_header.nh_flags &= NS_STANDALONE|NS_HASCHILD;
		owned_ = true;
	}
}

void *IndexEntry::allocate(size_t size)
{
	return new char[size];
}

void IndexEntry::deallocate(void *p)
{
	delete [] (char*)p;
}

bool IndexEntry::operator<(const IndexEntry &o) const
{
	if(docid_ != o.docid_)
		return docid_ < o.docid_;

	bool specified = isSpecified(NODE_ID);
	if(specified != o.isSpecified(NODE_ID))
		return specified;

	if(!specified) return false;

	return NsUtil::nsCompareIds(&node_->nd_header.nh_id,
				    &o.node_->nd_header.nh_id) < 0;
}

bool IndexEntry::isSpecified(Info info) const
{
	return isSpecified(format_, info);
}

bool IndexEntry::isSpecified(Format format, Info info)
{
	return indexFormats_[format][info];
}

const bool IndexEntry::indexFormats_[IndexEntry::KNOWN_FORMATS][IndexEntry::INFO_MAX] = {
	/*                NODE_ID,     NODE_LEVEL,      */
	/*                     L_CHILD_ID,    PARENT_ID */
	/* D_FORMAT    */ {false, false, false, false},
	/* DSEL_FORMAT */ {true,  true,  true,  false}
};
