//
// See the file LICENSE for redistribution information.
//
// Copyright (c) 2002-2005
//	Sleepycat Software.  All rights reserved.
//
// $Id: DocumentDatabase.cpp,v 1.22 2005/04/05 16:44:00 bostic Exp $
//

#include <sstream>

#include <dbxml/XmlException.hpp>
#include "DocumentDatabase.hpp"
#include "ScopedDbt.hpp"
#include "Container.hpp"
#include "OperationContext.hpp"
#include "Document.hpp"
#include "Cursor.hpp"
#include "UpdateContext.hpp"
#include "nodeStore/NsXercesIndexer.hpp"
#include "db_utils.h"

using namespace DbXml;
using namespace std;
static const char *document_name = "document";

DocumentDatabase::DocumentDatabase(DbEnv *env, Transaction *txn,
				   const std::string &name,
				   XmlContainer::ContainerType type,
				   u_int32_t pageSize, u_int32_t flags,
				   int mode)
	: environment_(env),
	  containerName_(name),
	  content_(env, name, "content_", document_name,
		   pageSize, flags & DB_XA_CREATE),
	  secondary_(env, name, document_name, 0, pageSize,
		     flags & DB_XA_CREATE),
	  type_(type)
{
	flags &= ~DB_XA_CREATE;
	int err = 0;
	try {
		// Open the Db objects
		if (type == XmlContainer::WholedocContainer) {
			err = content_.open(txn, DB_BTREE, flags, mode);
		}
		if (err == 0)
			err = secondary_.open(txn, false, flags, mode);
	} catch (DbException &e) {
		if(e.get_errno() == EEXIST) {
			throw XmlException(XmlException::CONTAINER_EXISTS,
					   e.what());
		} else {
			throw XmlException(e);
		}
	}
	if(err == EEXIST) {
		throw XmlException(XmlException::CONTAINER_EXISTS,
				   db_strerror(err));
	} else if(err != 0) {
		throw XmlException(err);
	}
}

DocumentDatabase::~DocumentDatabase()
{
	// nothing to do
	
}

int DocumentDatabase::getContent(OperationContext &context,
				 Document *document, u_int32_t flags) const
{
	DbtOut *data = new DbtOut();
	int err = 0;
	try {
		document->getID().setDbtFromThis(context.key());
		// NOTE: db_.get() does not throw on DB_NOTFOUND, but
		// try/catch anyway...
		err = content_.getDb().get(context.db_txn(),
					   &context.key(), data, flags);
	} catch (...) {
		delete data;
		throw; // re-throw
	}

	if(err == 0 && (data->get_size() != 0)) {
		document->setContentAsDbt(&data); // Note: consumes data
	} else {
		delete data;
	}
	if (err == DB_NOTFOUND)
		err = 0; // allow no-content documents
	return err;
}

// object is donated to caller
int DocumentDatabase::createDocumentCursor(
	Transaction *txn, scoped_ptr<DocumentCursor> &cursor) const
{
	DocumentDatabaseCursor *newCursor =
		new DocumentDatabaseCursor(txn, content_);
	cursor.reset(newCursor);
	return 0;
}

u_int32_t DocumentDatabase::getPageSize() const
{
	return content_.getPageSize();
}

unsigned long DocumentDatabase::getNumberOfPages() const
{
	return content_.getNumberOfPages();
}

// Remove the ID -- cannot "remove" an ID.  They are not
// reusable, but implement the call, just in case that ever
// changes.
int DocumentDatabase::deleteID(OperationContext &context, const ID& oldId)
{
	return 0; // this is a no-op; IDs are never reused
}

int DocumentDatabase::addContentAndIndex(Document &document,
					 UpdateContext &context,
					 Container &container,
					 KeyStash &stash)
{
	int err = 0;
	OperationContext &oc = context.getOperationContext();

	// If this document's content is a stream, convert it to a dbt,
	// because streams can only be used once... - jpcs
	if(document.getDefinitiveContent() == Document::INPUTSTREAM) {
		document.getContentAsDbt();
	}

	// Generate a unique ID for the Document, if not done by caller
	if (document.getID() == 0) {
		err = container.getConfigurationDB()->
			generateID(oc.txn(), document.getIDToSet());
		if (err)
			return err;
	}
	
	// Index the document
	context.getIndexer().indexDocument(context.getIndexSpecification(),
					   document, true, stash);

	Dbt *dbt = (Dbt*) document.getContentAsDbt();
	// Allow no-content documents; they have no entry in
	// the content database
	if (dbt && (dbt->get_size() != 0)) {
		document.getID().setDbtFromThis(oc.key());
		err = content_.getDb().put(oc.db_txn(), &oc.key(),
					   dbt, 0);
	}
	if(err == 0) document.setContentModified(false);
	return err;
}

int DocumentDatabase::updateContentAndIndex(Document &new_document,
					    UpdateContext &context,
					    KeyStash &stash)
{
	OperationContext &oc = context.getOperationContext();
	ID id = new_document.getID();
	Indexer &indexer = context.getIndexer();
	IndexSpecification &index = context.getIndexSpecification();
	int err = 0;
	bool resetId = false;

	// NOTE: consider adding DB_RMW if transacted.
	
	// Check to see if the old document exists, first
	// If ID is non-zero, let's trust it.  If not, get by name.
	// Retrieve the old document
	XmlDocument old_document = indexer.getContainer()->
		getManager().createDocument();

	if (id == 0) {
		// will throw if name is bad or doc doesn't exist
		err = indexer.getContainer()->getDocument(
			oc, new_document.getName(), old_document, DBXML_LAZY_DOCS);
		if (err == 0) {
			id = ((Document&)old_document).getID();
			new_document.getIDToSet() = id;
			resetId = true;
			// clear modified flag if set on name
			const_cast<Document*>(&new_document)->
				clearModified(Name(metaDataName_uri_name));
		}
	} else {
		err = indexer.getContainer()->getDocument(
			oc, id, old_document, DBXML_LAZY_DOCS);
	}
	if(err != 0) return err;
	
	// If this document's content is a stream, convert it to a dbt,
	// because streams can only be used once... - jpcs
	if(new_document.getDefinitiveContent() == Document::INPUTSTREAM) {
		new_document.getContentAsDbt();
	}

	// Index the new document
	indexer.indexMetaData(index, new_document, stash, true);
	if(new_document.isContentModified()) {
		scoped_ptr<NsEventSource8> source(new_document.getContentAsSAX(oc.txn(), true));
		indexer.indexContent(index, id,
				     source.get(), stash);
	}

	// Set the modified flags of the old document to the same as the
	// new document, so that when we index, we only generate keys for
	// things that have actually changed.
	MetaData::const_iterator end = new_document.metaDataEnd();
	for(MetaData::const_iterator i = new_document.metaDataBegin();
	    i != end; ++i) {
		if((*i)->isModified()) {
			const MetaDatum *md = ((Document&)old_document)
				.getMetaDataPtr((*i)->getName());
			if(md != 0) const_cast<MetaDatum*>(md)->setModified(true);
		}
	}

	// Remove the index keys for the old document
	IndexSpecification delete_index(index);
	delete_index.set(Index::INDEXER_DELETE);
	indexer.indexMetaData(delete_index, old_document, stash, true);
	if(new_document.isContentModified()) {
		scoped_ptr<NsEventSource8>
			source(((Document&)old_document).getContentAsSAX(oc.txn(), false));
		indexer.indexContent(delete_index, id,
				     source.get(), stash);
	}

	// Update the content
	if(new_document.isContentModified()) {
		OperationContext &oc = context.getOperationContext();
		id.setDbtFromThis(oc.key());
		err = content_.getDb().put(
			oc.db_txn(), &oc.key(),
			(Dbt*)new_document.getContentAsDbt(), 0);
	}

	if(err == 0) new_document.setContentModified(false);
	if (resetId)
		new_document.getIDToSet() = 0;
	return err;
}

int DocumentDatabase::removeContentAndIndex(const Document &document,
					    UpdateContext &context,
					    KeyStash &stash)
{
	// Index the document
	IndexSpecification &index = context.getIndexSpecification();
	index.set(Index::INDEXER_DELETE);
	context.getIndexer().indexDocument(index, document, false, stash);

	// Delete the content
	OperationContext &oc = context.getOperationContext();
	deleteID(oc, document.getID()); // a no-op
	document.getID().setDbtFromThis(oc.key());
	int err = content_.getDb().del(oc.db_txn(), &oc.key(), 0);
	
	return err;
}

int DocumentDatabase::getAllMetaData(OperationContext &context,
				     DictionaryDatabase *dictionary,
				     Document *document,
				     u_int32_t flags) const
{
	int err = 0;
	u_int32_t orig_flags = flags;
	const ID &did = document->getID();

	//
	// Read each of the meta-data items from the document secondary
	// database.  Content is no longer considered metadata
	//
	Cursor cursor(environment_,
		      const_cast<SecondaryDatabase*>(&secondary_)->getDb(),
		      context.txn(), CURSOR_READ);
	flags |= DB_SET_RANGE;
	bool done = false;
	while (!done) {
		did.setDbtFromThis(context.key());
		DbtIn none;
		none.set_flags(DB_DBT_PARTIAL); // Don't pull back the data.
		err = cursor.get(&context.key(), &none, flags);
		if (err == 0) {
			ID db_did, db_nid;
			XmlValue::Type type;
			MetaDatum::decodeKeyDbt(context.key(), db_did,
						db_nid, type);
			if (did == db_did) {
				Name name;
				err = dictionary->lookupName(context, db_nid, name);
				if(err == 0 && !document->containsMetaData(name)) {
					DbtOut *data = new DbtOut();
					err = cursor.get(&context.key(), data,
							 DB_CURRENT);
					if(err == 0) document->setMetaData(
						name, type, &data,
						/*modified*/false);
					delete data; // If not consumed by setThis..
				}
				flags = orig_flags | DB_NEXT;
			} else {
				err = 0;
				done = true;
			}
		} else if (err == DB_NOTFOUND) {
			err = 0;
			done = true;
		} else {
			done = true;
		}
	}
	return err;
}

int DocumentDatabase::getMetaData(OperationContext &context,
				  DictionaryDatabase *dictionary,
				  const Name &name,
				  const ID &did, Document *document,
				  u_int32_t flags) const
{
	ID nid;
	int err = dictionary->lookupName(context, name,
					 nid, /*define=*/false);
	if(err == 0) {
		Cursor cursor(
			environment_,
			const_cast<SecondaryDatabase*>(&secondary_)->getDb(),
			context.txn(), CURSOR_READ);
		MetaDatum::setKeyDbt(did, nid, XmlValue::NONE, context.key());
		DbtIn none;
		none.set_flags(DB_DBT_PARTIAL); // Don't pull back the data.
		err = cursor.get(&context.key(), &none, flags | DB_SET_RANGE);
		if (err == 0) {
			ID db_did, db_nid;
			XmlValue::Type type;
			MetaDatum::decodeKeyDbt(context.key(), db_did,
						db_nid, type);
			if(db_did == did && db_nid == nid) {
				DbtOut *data = new DbtOut();
				err = cursor.get(&context.key(), data, DB_CURRENT);
				if(err == 0)
					document->setMetaData(
						name, type,
						&data, /*modified*/false);
				delete data; // If not consumed by setThisFromDbt
			} else {
				return DB_NOTFOUND;
			}
		}
	}

	return err;
}

int DocumentDatabase::addMetaData(OperationContext &oc,
				  DictionaryDatabase *dictionary,
				  Document &document)
{
	int err = 0;
	MetaData::const_iterator end = document.metaDataEnd();
	MetaData::const_iterator i;
	for (i = document.metaDataBegin(); err == 0 && i != end; ++i) {
		ID nid;
		err = dictionary->lookupName(oc,
					     (*i)->getName(),
					     nid, /*define=*/true);
		if(err == 0) {
			DbtIn value;
			MetaDatum::setKeyDbt(document.getID(),
					     nid, (*i)->getType(),
					     oc.key());
			(*i)->setValueDbtFromThis(value);
			// could throw on error
			err = secondary_.put(oc.txn(), &oc.key(),
					     &value, 0);
		}
	}
	if(err == 0)
		for(i = document.metaDataBegin(); i != end; ++i)
			(*i)->setModified(false);
	return err;
}

int DocumentDatabase::updateMetaData(OperationContext &oc,
				     DictionaryDatabase *dictionary,
				     Document &document)
{
	int err = 0;
	MetaData::const_iterator end = document.metaDataEnd();
	MetaData::const_iterator i;
	std::vector<ID> toRemove;
	for(i = document.metaDataBegin(); err == 0 && i != end; ++i) {
		if((*i)->isModified()) {
			ID nid;
			err = dictionary->lookupName(oc,
						     (*i)->getName(),
						     nid, /*define=*/true);
			if(err == 0) {
				if ((*i)->isRemoved())
					toRemove.push_back(nid);
				else {
					DbtIn value;
					MetaDatum::setKeyDbt(document.getID(),
							     nid,
							     (*i)->getType(),
							     oc.key());
					(*i)->setValueDbtFromThis(value);
					// could throw on error
					err = secondary_.put(oc.txn(),
							     &oc.key(),
							     &value, 0);
				}
			}
		}
	}
	if (toRemove.size() > 0) {
		err = removeMetaData(oc, document.getID(),
				     &toRemove);
	}
	if(err == 0)
		for(i = document.metaDataBegin(); i != end; ++i)
			(*i)->setModified(false);
	return err;
}

static bool
idInList(const std::vector<ID> &list, const ID &id)
{
	std::vector<ID>::const_iterator it = list.begin();
	while (it != list.end()) {
		if (*it == id)
			return true;
		it++;
	}
	
	return false;
}

// if toRemove is non-null, it specifies a list of Name IDs
// to remove; otherwise remove all metadata for the target document
int DocumentDatabase::removeMetaData(OperationContext &oc,
				     const ID &id,
				     std::vector<ID> *toRemove)
{
	Cursor cursor(environment_, secondary_.getDb(), oc.txn(),
		      CURSOR_WRITE);

	DbtIn none;
	none.set_flags(DB_DBT_PARTIAL); // Don't pull back the data.

	id.setDbtFromThis(oc.key());

	ID db_id;
	int err = cursor.get(&oc.key(), &none, DB_SET_RANGE);
	while(err == 0) {
		if (toRemove) {
			ID nm_id;
			XmlValue::Type type;
			MetaDatum::decodeKeyDbt(oc.key(), db_id, nm_id, type);
			if ((id == db_id) && idInList(*toRemove, nm_id))
				cursor.del(0);
		} else {
			db_id.setThisFromDbt(oc.key());
			if (id == db_id)
				cursor.del(0);
		}
		if (id != db_id) // done with document?
			break;
		err = cursor.get(&oc.key(), &none, DB_NEXT);
	}

	if(err == DB_NOTFOUND) {
		err = 0;
	}

	return err;
}

int DocumentDatabase::dump(DbEnv *env, const std::string &name,
			   XmlContainer::ContainerType type,
			   std::ostream *out)
{
	DbWrapper content(env, name, "content_", document_name, 0, 0);
	SecondaryDatabase secondary(env, name, document_name, 0,
				    0, 0);
	int err = 0;
	if (type == XmlContainer::WholedocContainer) {
		err = Container::writeHeader(content.getDatabaseName(), out);
		if(err == 0)
			err = content.dump(out);
	}
	if(err == 0)
		err = Container::writeHeader(secondary.getDatabaseName(), out);
	if(err == 0)
		err = secondary.dump(out);

	return err;
}

int DocumentDatabase::load(DbEnv *env, const std::string &name,
			   XmlContainer::ContainerType type,
			   std::istream *in, unsigned long *lineno)
{
	DbWrapper content(env, name, "content_", document_name, 0, 0);
	SecondaryDatabase secondary(env, name, document_name, 0, 0, 0);

	int err = 0;
	if (type == XmlContainer::WholedocContainer) {
		// Load primary
		err = Container::verifyHeader(content.getDatabaseName(), in);
		if(err != 0) {
			ostringstream oss;
			oss << "DocumentDatabase::load() invalid database dump file loading '" << name << "'";
			Log::log(env, Log::C_CONTAINER, Log::L_ERROR, oss.str().c_str());
		} else {
			err = content.load(in, lineno);
		}
	}

	// Load secondary
	if(err == 0) {
		err = Container::verifyHeader(secondary.getDatabaseName(), in);
		if(err != 0) {
			ostringstream oss;
			oss << "DocumentDatabase::load() invalid database dump file loading '" << name << "'";
			Log::log(env, Log::C_CONTAINER, Log::L_ERROR, oss.str().c_str());
		} else {
			err = secondary.load(in, lineno);
		}
	}

	return err;
}

int DocumentDatabase::verify(DbEnv *env, const std::string &name,
			     XmlContainer::ContainerType type,
			     std::ostream *out, u_int32_t flags)
{
	DbWrapper content(env, name, "content_", document_name, 0, 0);
	SecondaryDatabase secondary(env, name, document_name, 0, 0, 0);

	int err = 0;
	if (type == XmlContainer::WholedocContainer) {
		if(flags & DB_SALVAGE)
			err = Container::writeHeader(content.getDatabaseName(),
						     out);
		if(err == 0)
			err = content.verify(out, flags);
	}
	if(err == 0 && (flags & DB_SALVAGE))
		err = Container::writeHeader(secondary.getDatabaseName(), out);
	if(err == 0)
		err = secondary.verify(out, flags);
	return err;
}

DocumentDatabaseCursor::DocumentDatabaseCursor(Transaction *txn,
					       DbWrapper &db) :
	cursor_(db.getEnvironment(), db.getDb(), txn, CURSOR_READ),
	done_(false) {
	data_.set_flags(DB_DBT_PARTIAL); // only want keys (for now)
}

int DocumentDatabaseCursor::next(ID &id) {
	int err = 0;
	if (done_) {
		id = 0;
	} else {
		err = cursor_.get(&key_, &data_, DB_NEXT_NODUP); // no throw
		if (err == 0) {
			// skip DbSequence key -- it's the only one
			// with len > 4. Can only recurse once
			if (key_.get_size() > sizeof(u_int32_t))
				return next(id);
			id.setThisFromDbt(key_);
		}
		if (err == DB_NOTFOUND || err == DB_KEYEMPTY) {
			err = 0;
			done_ = true;
			id = 0;
		}
	}
	return err;
}

void DocumentDatabase::upgrade(int saved_version, int current_version)
{
	if (saved_version == VERSION_20) {
		// byte-swap name ids in document metadata keys.
		// iterate through all records in secondary.
		// the key is: docId,nodeId
		Dbt key;
		Dbt data;
		Cursor curs(environment_, secondary_.getDb(),
			    (Transaction*)0, DbXml::CURSOR_WRITE);
		int ret = 0;
		while ((ret = curs.get(&key, &data, DB_NEXT)) == 0) {
			u_int32_t *p = reinterpret_cast<u_int32_t*>(key.get_data());
			curs.del(0);
			++p;
			M_32_SWAP(*p);
			int putRet = curs.put(&key, &data, DB_KEYFIRST);
			if (putRet != 0) {
				std::cerr <<"put returned error: " << putRet << std::endl;
			}
		}
	}
}
