//
// See the file LICENSE for redistribution information.
//
// Copyright (c) 2002-2005
//	Sleepycat Software.  All rights reserved.
//
// $Id: Cursor.hpp,v 1.32 2005/04/05 16:43:59 bostic Exp $
//

#ifndef __CURSOR_HPP
#define	__CURSOR_HPP

#include "DbWrapper.hpp"
#include "ScopedDbt.hpp"

namespace DbXml
{

typedef enum CursorType { CURSOR_READ, CURSOR_WRITE } CursorType;

/// Provides exception safety for a Berkeley Db Cursor.
class Cursor
{
public:
	Cursor(DbEnv *environment, Db &db, Transaction *txn, CursorType type);
	~Cursor()
	{
		if (error_ == 0)
			dbc_->close();
	}
	int error() const
	{
		return error_;
	}
	int get(Dbt *key, Dbt *data, u_int32_t flags)
	{
		return dbc_->get(key, data, flags);
	}
	int put(Dbt *key, Dbt *data, u_int32_t flags)
	{
		return dbc_->put(key, data, flags);
	}
	int del(u_int32_t flags)
	{
		return dbc_->del(flags);
	}
private:
	// no need for copy and assignment
	Cursor(const Cursor&);
	Cursor &operator = (const Cursor &);

	Dbc *dbc_;
	int error_;
};

// Wraps a Cursor for a DocumentDatabase
// Virtual behavior is implemented by specific DocumentDatabase impl.
// Construction is via DocumentDatabase::createDocumentCursor()
class DocumentCursor
{
public:
	virtual ~DocumentCursor() {}
	virtual int first(ID &id) = 0; // done when id == 0
	virtual int next(ID &id) = 0; // done when id == 0
};

/// Wraps a Cursor, for a Secondary Database.
class SecondaryCursor
{
public:
	SecondaryCursor(SecondaryDatabase &db, Transaction *txn,
			DbWrapper::Operation operation, const Key *k1,
			const Syntax *syntax);
	SecondaryCursor(SecondaryDatabase &db, Transaction *txn,
			DbWrapper::Operation gto, const Key *k1,
			DbWrapper::Operation lto, const Key *k2,
			const Syntax *syntax);
	~SecondaryCursor();
	int error() const
	{
		return cursor_.error();
	}

	/// Done when id==0
	int first(IndexEntry &ie);
	/// Done when id==0
	int next(IndexEntry &ie);

private:
	// no need for copy and assignment
	SecondaryCursor(const SecondaryCursor&);
	SecondaryCursor & operator = (const SecondaryCursor &);

	int doNext(IndexEntry &ie, int flags);

	DbEnv *environment_;
	const Syntax *syntax_; // Provides functions for comparing keys.
	DbWrapper::Operation operation_; // Operation passed through the ctor
	// Ranges: The upper bound operation. GTX|GTE
	DbWrapper::Operation greaterThanOperation_;
	// Ranges: The lower bound operation. LTX|LTE
	DbWrapper::Operation lessThanOperation_; 
	Cursor cursor_;
	// The original key we are looking for.
	// Only for prefix, and inequality searches
	DbtOut originalKey_;
	DbtOut key_;  // The key we pass into get(). Range: The start of the range.
	DbtOut key2_; // The key we pass into get(). Range: The end of the range.
	DbtOut data_; // The data we get back from get().
	bool done_;
};

}

#endif

