//
// See the file LICENSE for redistribution information.
//
// Copyright (c) 2002-2005
//	Sleepycat Software.  All rights reserved.
//
// $Id: Cursor.cpp,v 1.45 2005/04/05 16:43:59 bostic Exp $
//

#include "dbxml_config.h"
#include "dbxml/XmlPortability.hpp"
#include "Cursor.hpp"
#include "Key.hpp"
#include "Log.hpp"
#include "IndexEntry.hpp"

#include <cassert>
#include <sstream>

using namespace DbXml;

// Cursor
Cursor::Cursor(DbEnv *environment, Db &db, Transaction *txn, CursorType type)
	: dbc_(0),
	error_( -1)
{
	u_int32_t flags = 0;
	if (environment != 0 && type==CURSOR_WRITE) {
		// If we're in a CDS environment then the cursor used to
		// delete the keys must be created with the DB_WRITECURSOR flag.
		//
		if (environment->get_DB_ENV()->flags & DB_ENV_CDB) {
			flags = DB_WRITECURSOR;
		}
	}
	error_ = db.cursor(Transaction::toDbTxn(txn), &dbc_, flags);
}

// SecondaryCursor

SecondaryCursor::SecondaryCursor(SecondaryDatabase &db, Transaction *txn, DbWrapper::Operation operation, const Key *k1, const Syntax *syntax)
	: environment_(db.getEnvironment()),
	syntax_(syntax),
	operation_(operation),
	greaterThanOperation_(DbWrapper::NONE),
	lessThanOperation_(DbWrapper::NONE),
	cursor_(db.getEnvironment(), db.getDb(), txn, CURSOR_READ),
	done_(false)
{
	if (operation_ != DbWrapper::ALL) {
		k1->setDbtFromThis(key_);
		originalKey_.set(key_.get_data(), key_.get_size());
	}
}

SecondaryCursor::SecondaryCursor(SecondaryDatabase &db, Transaction *txn, DbWrapper::Operation gto, const Key *gtk, DbWrapper::Operation lto, const Key *ltk, const Syntax *syntax)
	: environment_(db.getEnvironment()),
	syntax_(syntax),
	operation_(DbWrapper::RANGE),
	greaterThanOperation_(gto),
	lessThanOperation_(lto),
	cursor_(db.getEnvironment(), db.getDb(), txn, CURSOR_READ),
	done_(false)
{
	assert(lto == DbWrapper::LTX || lto == DbWrapper::LTE);
	assert(gto == DbWrapper::GTX || gto == DbWrapper::GTE);
	gtk->setDbtFromThis(key_);
	ltk->setDbtFromThis(key2_);
}

SecondaryCursor::~SecondaryCursor()
{
	environment_ = 0;
	syntax_ = 0;
}

int SecondaryCursor::first(IndexEntry &ie)
{
	int flags = 0;
	int err = 0;
	DbWrapper::Operation operation = (operation_ == DbWrapper::RANGE ? greaterThanOperation_ : operation_);	// For a RANGE we start with the greaterThanOperation_.
	switch (operation) {
	case DbWrapper::ALL:
		flags = DB_FIRST;
		break;
	case DbWrapper::EQUALITY:
		flags = DB_SET;
		break;
	case DbWrapper::LTX:  // a<b
		// Skip the first matching key.
		flags = DB_SET_RANGE;
		err = cursor_.get(&key_, &data_, flags); // no throw for NOTFOUND and KEYEMPTY
		if (err == DB_NOTFOUND) {
			err = 0;
			flags = DB_LAST;
		} else {
			flags = DB_PREV;
		}
		break;
	case DbWrapper::LTE:  // a<=b
		// LTE is a bit special, it behaves as equality until there are
		// no more equal keys, then switches to LTX. See the switcheroo
		// below in doNext.
		flags = DB_SET;
		break;
	case DbWrapper::GTX:  // a>b
		// Skip the first matching duplicate keys.
		flags = DB_SET;
		err = cursor_.get(&key_, &data_, flags); // no throw for NOTFOUND and KEYEMPTY
		if (err == DB_NOTFOUND) {
			err = 0;
			flags = DB_SET_RANGE;
		} else {
			flags = DB_NEXT_NODUP;
		}
		break;
	case DbWrapper::GTE:  // a>=b
		flags = DB_SET_RANGE;
		break;
	case DbWrapper::PREFIX:
		flags = DB_SET_RANGE; // Return both the key and data on each get.
		break;
	case DbWrapper::SUBSTRING:
		// This just won't happen.
		err = DB_NOTFOUND;
		assert(0);
		break;
	}
	if (err == 0) {
		err = doNext(ie, flags);
	} else {
		if (err == DB_NOTFOUND || err == DB_KEYEMPTY) {
			err = 0;
		}
		done_ = true;
		ie.reset();
	}
	return err;
}

int SecondaryCursor::next(IndexEntry &ie)
{
	int err = 0;
	if (done_) {
		ie.reset();
	} else {
		int flags = 0;
		switch (operation_) {
		case DbWrapper::ALL:
			flags = DB_NEXT;
			break;
		case DbWrapper::EQUALITY:
			flags = DB_NEXT_DUP;
			break;
		case DbWrapper::LTX:  // a<b
			flags = DB_PREV;
			break;
		case DbWrapper::LTE:  // a<=b
			// LTE is a bit special, it behaves as equality until there are
			// no more equal keys, then switches to LTX. See the switcheroo
			// below in doNext.
			flags = DB_NEXT_DUP;
			break;
		case DbWrapper::GTX:  // a>b
		case DbWrapper::GTE:  // a>=b
			flags = DB_NEXT;
			break;
		case DbWrapper::RANGE:
			flags = DB_NEXT;
			break;
		case DbWrapper::PREFIX:
			// We make a copy of the key because dbc->get() will return the actual key.
			flags = DB_NEXT;
			key_.set(originalKey_.get_data(), originalKey_.get_size());
			break;
		case DbWrapper::NONE:
		case DbWrapper::SUBSTRING:
			// This just won't happen.
			err = DB_NOTFOUND;
			assert(0);
			break;
		}
		err = doNext(ie, flags);
	}
	return err;
}

int SecondaryCursor::doNext(IndexEntry &ie, int flags)
{
	// jcm - We could use the bulk key interface here for a performance boost.
	int err = cursor_.get(&key_, &data_, flags);
	if (err == 0) {
		switch (operation_) {
		case DbWrapper::PREFIX:
			if(key_.get_size() < originalKey_.get_size() || memcmp(originalKey_.get_data(), key_.get_data(), originalKey_.get_size()) != 0) {
				done_ = true;
				ie.reset();
			}
			break;
		case DbWrapper::LTX:  // a<b
		case DbWrapper::GTX:  // a>b
		case DbWrapper::GTE:  // a>=b
			{
				if (syntax_ == 0)
				{
					std::ostringstream oss;
					oss << "No syntax provided for secondary cursor (<,>,>=)";
					Log::log(environment_, Log::C_QUERY, Log::L_ERROR, 0, oss.str().c_str());
					done_ = true;
					ie.reset();
					assert(0);
				} else
				{
					char *p = (char*)originalKey_.get_data();
					Index index;
					index.setFromPrefix(*p);
 					size_t skl = syntax_->structureKeyLength(index, originalKey_.get_size());
					if (::memcmp(originalKey_.get_data(), key_.get_data(), skl) != 0) // Check the Prefix and VIDs are the same.
					{
						done_ = true;
						ie.reset();
					}
				}
			}
			break;
		case DbWrapper::RANGE: {
				if (syntax_ == 0) {
					std::ostringstream oss;
					oss << "No syntax provided for secondary cursor (range).";
					Log::log(environment_, Log::C_QUERY, Log::L_ERROR, 0, oss.str().c_str());
					done_ = true;
					ie.reset();
					assert(0);
				} else {
					DbWrapper::bt_compare_fn compare = syntax_->get_bt_compare();
					int r = compare(0, &key_, &key2_);
					if (!((lessThanOperation_ == DbWrapper::LTE && r <= 0) || (lessThanOperation_ == DbWrapper::LTX && r < 0))) {
						done_ = true;
						ie.reset();
					}
				}
			}
			break;
		default:  // keep compilers quiet
			break;
		}
		if (!done_) {
			ie.setThisFromDbt(data_);
		}
	} else {
		if (err == DB_NOTFOUND || err == DB_KEYEMPTY) {
			if (operation_ == DbWrapper::LTE) {
				// Once LTE has exhausted the equal keys it starts on the
				// less than keys.
				operation_ = DbWrapper::LTX;
				err = first(ie);
			} else {
				done_ = true;
				ie.reset();
				err = 0;
			}
		} else {
			done_ = true;
			ie.reset();
		}
	}
	return err;
}

