# See the file LICENSE for redistribution information.
#
# Copyright (c) 2000-2003
#	Sleepycat Software.  All rights reserved.
#
# $Id: xml003.tcl,v 1.22 2003/12/09 14:02:44 merrells Exp $
#
# TEST	xml003
# TEST	Transaction testing

proc xml003 { {args ""} } {
	puts "\nXml003: Transaction testing ($args)"
	source ./include.tcl

	# Determine whether procedure has been called within an environment.
	# Skip test if we are given an env.  It needs its own. 
	set eindex [lsearch -exact $args "-env"]
	set tnum 3
	set basename $tnum

	# oargs is used for arguments to XmlContainer open
	set oargs [eval {set_openargs} $args]
	set oargs_saved $oargs
	# eargs is used for argument to berkdb_env
	set eargs " -create "
	if { [lsearch -exact $args thread] >= 0 } {
		append eargs " -thread "
	}

	if { $eindex != -1 } {
		incr eindex
		set env [lindex $args $eindex]
		puts "Skipping xml003 for env $env"
		return
	}

	# Set up test env for explicit txn testing. 
	env_cleanup $testdir
	set env [eval {berkdb_env} $eargs -txn -mode 0644 -home $testdir]
	dbxml_error_check_good open_txn_env [is_valid_env $env] TRUE

	puts "\tXml003.1: basic commit and abort tests"

	xml003_1_1_1 $env $basename $oargs
	xml003_1_1_2 $env $basename $oargs
	xml003_1_2 $env $basename $oargs
	xml003_1_3 $env $basename $oargs
	xml003_1_4 $env $basename $oargs
	xml003_1_5_1 $env $basename $oargs
	xml003_1_5_2 $env $basename $oargs
	xml003_1_6_1 $env $basename $oargs
	xml003_1_6_2 $env $basename $oargs
	xml003_1_7 $env $basename $oargs
	xml003_1_8 $env $basename $oargs
	xml003_1_9 $env $basename $oargs

	dbxml_error_check_good env_close [$env close] 0

	puts "\tXml003.2: auto_commit tests"
	# Add auto_commit flag to oargs for xml003_2 tests.
	set oargs [expr $oargs + $DB_AUTO_COMMIT]

	# Set up non-txn env for test 3.2.1   
	env_cleanup $testdir
	set env [eval {berkdb_env_noerr} $eargs -mode 0644 -home $testdir]
	dbxml_error_check_good non_txn_env [is_valid_env $env] TRUE

	xml003_2_1 $env $basename $oargs	

	dbxml_error_check_good env_close [$env close] 0
	
	# Set up a new txn env for the bulk of auto_commit testing.
	env_cleanup $testdir
	set env [eval {berkdb_env} $eargs -txn -mode 0644 -home $testdir]
	dbxml_error_check_good open_txn_env [is_valid_env $env] TRUE

	xml003_2_2 $env $basename $oargs
	xml003_2_3 $env $basename $oargs
	xml003_2_4 $env $basename $oargs
	xml003_2_5 $env $basename $oargs
	xml003_2_6_1 $env $basename $oargs
	xml003_2_6_2 $env $basename $oargs
	xml003_2_7_1 $env $basename $oargs
	xml003_2_7_2 $env $basename $oargs
	xml003_2_8 $env $basename $oargs
	xml003_2_9 $env $basename $oargs

	dbxml_error_check_good env_close [$env close] 0

	puts "\tXml003.3: Transactionally linked DB-XML and BDB records."
	# Remove $DB_AUTO_COMMIT flag for xml003_3 tests.
	set oargs $oargs_saved
	# Open a transactional env.
	set env [eval {berkdb_env} $eargs -txn -mode 0644 -home $testdir]
	dbxml_error_check_good open_txn_env [is_valid_env $env] TRUE

	xml003_3 $env $basename $oargs

	dbxml_error_check_good env_close [$env close] 0
	return
}

proc xml003_1_1_1 { {env "NULL"} {basename $tnum} oargs } {
	puts "\t\tXml003.1.1.1: Container open"
	source ./include.tcl

	new XmlContainer container $env "$basename.1.1.1.dbxml" 0

	foreach op "commit abort" {
		puts "\t\t\tTest $op"
		set txn [start_txn $env] 
		$container open $txn $oargs 0
		dbxml_error_check_good txn_$op [$txn $op] 0

		catch { $container close } ret

		if { $op == "abort" } {
			# If the open was aborted, the attempt to close should have 
			# failed.  Check for the appropriate error.
			dbxml_error_check_good open_abort \
			    [is_substr $ret "close() not allowed on closed XmlContainer"] 0
		} else {
			# The open was committed, so the attempt to close should have 
			# succeeded.
			dbxml_error_check_good open_commit [llength $ret] 0
		} 
	}
	delete container	

}

proc xml003_1_1_2 { {env "NULL"} {basename $tnum} oargs } {
	puts "\t\tXml003.1.1.2: Container exists"
	source ./include.tcl

	xml_cleanup $testdir $env
	new XmlContainer container $env "$basename.1.1.2.dbxml" 0
	# Make sure the file doesn't yet exist
	dbxml_error_check_good not_exists [$container exists NULL] 0

	set txn [start_txn $env] 
	$container open $txn $oargs 0
	# Test a transaction-protected existence test after an open
	dbxml_error_check_good exists [$container exists $txn] 1
	dbxml_error_check_good txn_commit [$txn commit] 0

	# The open was committed, so the attempt to close should succeed.
	catch { $container close } ret
	dbxml_error_check_good open_commit [llength $ret] 0

	# Make sure the file does now exist
	dbxml_error_check_good exists [$container exists NULL] 1

	delete container	
}

proc xml003_1_2 { {env "NULL"} {basename $tnum} oargs } {
	puts "\t\tXml003.1.2: addIndex"
	source ./include.tcl
	
	new XmlContainer container $env "$basename.1.2.dbxml" 0

	foreach op "abort commit" {
		puts "\t\t\tTest $op"
		set txn [start_txn $env] 
		$container open $txn $oargs 0
		dbxml_error_check_good open_commit [$txn commit] 0

		set uri www.uri.com
		set name foo
		set index node-element-presence-none

		set txn [start_txn $env]
		$container addIndex $txn $uri $name $index
		dbxml_error_check_good txn_$op [$txn $op] 0

		# Test that index spec was persisted.
		wrap XmlIndexSpecification indexes [$container getIndexSpecification NULL]
		set idecl [$indexes find $uri $name]
		delete indexes

		if { $op == "abort" } {
			# If addIndex was aborted, the index should not
			# have been found.
			dbxml_error_check_good declare_abort $idecl NULL
		} else {
			# If addIndex was committed, the index specification
			# should persist.
			dbxml_error_check_bad declare_commit $idecl NULL
			wrap XmlIndexDeclaration xid $idecl
			set retindex [$xid get_index]
			dbxml_error_check_good index $index $retindex
			delete xid
		}
		$container close
	}
	delete container	
}

proc xml003_1_3 { {env "NULL"} {basename $tnum} oargs } {
	puts "\t\tXml003.1.3: putDocument."
	source ./include.tcl
	
	new XmlContainer container $env "$basename.1.3.dbxml" 0

	foreach op "commit abort" {
		puts "\t\t\tTest $op"
		set txn [start_txn $env] 
		$container open $txn $oargs 0
		dbxml_error_check_good open_commit [$txn commit] 0

		new XmlDocument xd
		set data_in "<a><b>c</b></a>"
		$xd setContent $data_in

		set txn [start_txn $env]
		set id [$container putDocument $txn $xd]
		dbxml_error_check_good "put_$op" [$txn $op] 0
		delete xd

		catch {
			wrap XmlDocument xd [$container getDocument NULL $id]
			set data_out [$xd getContentAsString]
			delete xd
		} ret

		if { $op == "abort" } {
			# If the put was aborted, the attempt to get a document should
			# give a 'not found' error.
			dbxml_error_check_good check_abort \
			    [is_substr $ret "Document not found"] 1		
		} else {
			# If the put was committed, the retrieved data should match 
			# the loaded data.
			dbxml_error_check_good content_correct $data_in $data_out
		} 
		$container close
	}
	delete container
}

# Test 3.4 - state doesn't change in the case of an aborted getDocument
# so we are just testing that the interface doesn't create unexpected errors.
proc xml003_1_4 { {env "NULL"} {basename $tnum} oargs } {
	puts "\t\tXml003.1.4: Commit and abort on getDocument."
	source ./include.tcl
	xml_cleanup $testdir $env
	
	new XmlContainer container $env "$basename.1.4.dbxml" 0

	foreach op "commit abort" {
		puts "\t\t\tTest $op"
	
		set txn [start_txn $env] 
		$container open $txn $oargs 0
		dbxml_error_check_good open_commit [$txn commit] 0

		new XmlDocument xd
		set data_in "<a><b>c</b></a>"
		$xd setContent $data_in

		set txn [start_txn $env]
		set id [$container putDocument $txn $xd]
		dbxml_error_check_good open_commit [$txn commit] 0

		delete xd
		set txn [start_txn $env]
		wrap XmlDocument xd [$container getDocument $txn $id]
		dbxml_error_check_good "txn $op" [$txn $op] 0

		delete xd

		$container close
	}
	delete container
}

proc xml003_1_5_1 { {env "NULL"} {basename $tnum} oargs } {
	puts "\t\tXml003.1.5.1: Delete with XmlDocument object."
	source ./include.tcl
	xml_cleanup $testdir $env
	
	new XmlContainer container $env "$basename.1.5.1.dbxml" 0

	foreach op "commit abort" {
		puts "\t\t\tTest $op"
		set txn [start_txn $env] 
		$container open $txn $oargs 0
		dbxml_error_check_good open_commit [$txn commit] 0

		new XmlDocument xd
		set data_in "<a><b>c</b></a>"
		$xd setContent $data_in

		set txn [start_txn $env]
		set id [$container putDocument $txn $xd]
		dbxml_error_check_good open_commit [$txn commit] 0

		set txn [start_txn $env]
		catch {$container deleteDocument $txn $xd} ret
		dbxml_error_check_good delete_$op [$txn $op] 0
		delete xd
		
		set txn [start_txn $env]
		catch {
			wrap XmlDocument xd [$container getDocument $txn $id]
		} ret
		dbxml_error_check_good get_commit [$txn commit] 0

		if { $op == "abort" } {
			# If the delete was aborted, the 'get' should have been successful, 
			# and we can check the data.
			set data_out [$xd getContentAsString]
			dbxml_error_check_good content_correct $data_in $data_out
			delete xd			
		} else {
			# op was "commit".  Check for appropriate error on the getDocument.
			dbxml_error_check_good doc_deleted $ret "Error: Document not found."
		} 
		$container close
	}
	delete container
}

proc xml003_1_5_2 { {env "NULL"} {basename $tnum} oargs } {
	puts "\t\tXml003.1.5.2: Delete with document ID."
	source ./include.tcl
	xml_cleanup $testdir $env
	
	new XmlContainer container $env "$basename.1.5.2.dbxml" 0

	foreach op "commit abort" {
		puts "\t\t\tTest $op"
		set txn [start_txn $env] 
		$container open $txn $oargs 0
		dbxml_error_check_good open_commit [$txn commit] 0

		new XmlDocument xd
		set data_in "<a><b>c</b></a>"
		$xd setContent $data_in

		set txn [start_txn $env]
		set id [$container putDocument $txn $xd]
		dbxml_error_check_good "put_commit" [$txn commit] 0
		delete xd

		set txn [start_txn $env]
		catch {$container deleteDocumentByID $txn $id} ret
		dbxml_error_check_good "put_$op" [$txn $op] 0

		set txn [start_txn $env]
		catch {	
			wrap XmlDocument xd [$container getDocument $txn $id]
		} ret
		dbxml_error_check_good get_commit [$txn commit] 0

		if { $op == "abort" } {
			# If the delete was aborted, the 'get' should have been successful, 
			# and we can check the data.
			set data_out [$xd getContentAsString]
			dbxml_error_check_good content_correct $data_in $data_out
			delete xd			
		} else {
			# op was "commit".  Check for appropriate error on getDocument.
			dbxml_error_check_good doc_deleted $ret "Error: Document not found."
		}
		$container close
	}
	delete container
}

# Test 3.1.6 - state doesn't change in the case of an aborted queryWithXPath
# so we are just testing that the interface doesn't create unexpected errors.
proc xml003_1_6_1 { {env "NULL"} {basename $tnum} oargs } {
	puts "\t\tXml003.1.6.1: queryWithXPath using XPathExpression."
	source ./include.tcl
	xml_cleanup $testdir $env
	
	new XmlContainer container $env "$basename.1.6.1.dbxml" 0

	foreach op "commit abort" {
		puts "\t\t\tTest $op"
		set txn [start_txn $env] 
		$container open $txn $oargs 0
		dbxml_error_check_good open_commit [$txn commit] 0

		new XmlDocument xd
		new XmlQueryContext context
		set data_in "<a><b>c</b></a>"
		$xd setContent $data_in

		set txn [start_txn $env]
		set id [$container putDocument $txn $xd]
		dbxml_error_check_good "put_commit" [$txn commit] 0

		$context setReturnType $XmlQueryContext_ResultDocuments
		$context setEvaluationType $XmlQueryContext_Eager

		# Run transacted query and check return value
		new XmlValue xv
		set txn [start_txn $env]
		wrap XmlQueryExpression xe [$container parseXPathExpression $txn "/a/b" $context]
		wrap XmlResults xr [$container queryWithXPathExpression $txn $xe]
		dbxml_error_check_good txn_$op [$txn $op] 0
		$xr next $xv

		dbxml_error_check_good "check_contents" [$xv asString] [$xd getContent]
		dbxml_error_check_good "size_of_result_set" [$xr size] 1
		delete xr
		delete xv
		delete xd
		delete xe
		delete context

		$container close
		# Remove container so size of result set does not change.
		$container remove NULL
	}
	delete container
}

proc xml003_1_6_2 { {env "NULL"} {basename $tnum} oargs } {
	puts "\t\tXml003.1.6.2: queryWithXPath using XmlQueryContext."
	source ./include.tcl
	xml_cleanup $testdir $env
	
	new XmlContainer container $env "$basename.1.6.2.dbxml" 0

	foreach op "commit abort" {
		puts "\t\t\tTest $op"
		set txn [start_txn $env] 
		$container open $txn $oargs 0
		dbxml_error_check_good open_commit [$txn commit] 0

		new XmlQueryContext context
		new XmlDocument xd
		set data_in "<a><b>c</b></a>"
		$xd setContent $data_in

		set txn [start_txn $env]
		set id [$container putDocument $txn $xd]
		dbxml_error_check_good txn_commit [$txn commit] 0

		$context setReturnType $XmlQueryContext_ResultDocuments
		$context setEvaluationType $XmlQueryContext_Eager

		# Run transacted query and check return value
		new XmlValue xv
		set txn [start_txn $env]
		wrap XmlResults xr [$container queryWithXPath $txn "/a/b" $context]
		$xr next $xv
		dbxml_error_check_good txn_$op [$txn $op] 0

		dbxml_error_check_good "check_contents" [$xv asString] [$xd getContent]
		dbxml_error_check_good "size_of_result_set" [$xr size] 1

		delete xv
		delete xr
		delete xd
		delete context
		$container close
		# Remove container so size of result set does not change.		
		$container remove NULL
		}
	delete container
}

proc xml003_1_7 { {env "NULL"} {basename $tnum} oargs } {
	puts "\t\tXml003.1.7: Remove container"
	source ./include.tcl

	foreach op "commit abort" {
		xml_cleanup $testdir $env
		new XmlContainer container $env "$basename.1.7.dbxml" 0
		puts "\t\t\tTest $op"

		set txn [start_txn $env] 
		$container open $txn $oargs 0
		dbxml_error_check_good txn_commit [$txn commit] 0

		$container close

		set txn [start_txn $env]
		$container remove $txn
		dbxml_error_check_good "txn_$op" [$txn $op] 0

		if { $op == "abort" } {
			# Underlying files should still exist.
			dbxml_error_check_good "original primary_dict exists" \
			    [file exists $testdir/3.1.7.dbxml] 1
		} else {
			# $op was a commit, underlying files should be gone.
			dbxml_error_check_good "original primary_dict exists" \
			    [file exists $testdir/3.1.7.dbxml] 0
		}
		delete container	
	}
}

proc xml003_1_8 { {env "NULL"} {basename $tnum} oargs } {
	puts "\t\tXml003.1.8: Rename container"
	source ./include.tcl

	foreach op "commit abort" {
		puts "\t\t\tTest $op"
		xml_cleanup $testdir $env
		new XmlContainer container $env "$basename.1.8.dbxml" 0

		set txn [start_txn $env] 
		$container open $txn $oargs 0
		dbxml_error_check_good txn_commit [$txn commit] 0
		$container close

		set txn [start_txn $env]
		$container rename $txn "$basename.1.8_renamed.dbxml"
		dbxml_error_check_good "txn_$op" [$txn $op] 0

		if { $op == "abort" } {
			# Renamed file will not exist, and original file will.
			dbxml_error_check_good "renamed primary_dict exists" \
			    [file exists $testdir/3.1.8_renamed.dbxml] 0
			dbxml_error_check_good "original primary_dict exists" \
			    [file exists $testdir/3.1.8.dbxml] 1
		} else {
			# $op was a commit
			# Renamed file will exist, and original file will not.
			dbxml_error_check_good "renamed primary_dict exists" \
			    [file exists $testdir/3.1.8_renamed.dbxml] 1
			dbxml_error_check_good "original primary_dict exists" \
			    [file exists $testdir/3.1.8.dbxml] 0
		}
		delete container	
	}
}

proc xml003_1_9 { {env "NULL"} {basename $tnum} oargs } {
	puts "\t\tXml003.1.9: XmlResults: next"
	source ./include.tcl
	xml_cleanup $testdir $env
	new XmlContainer container $env "$basename.1.9.dbxml" 0

	foreach op "commit abort" {
		puts "\t\t\tTest $op"
		set txn [start_txn $env] 
		$container open $txn $oargs 0
		dbxml_error_check_good "txn_commit" [$txn commit] 0

		new XmlQueryContext context
		new XmlDocument xd
		$xd setContent "<a><b>c</b></a>"

		set txn [start_txn $env]
		$container putDocument $txn $xd
		dbxml_error_check_good "txn_commit" [$txn commit] 0
		delete xd		

		new XmlValue xv

		$context setReturnType $XmlQueryContext_ResultDocuments
		$context setEvaluationType $XmlQueryContext_Eager

		set txn [start_txn $env]
		wrap XmlResults xr [$container queryWithXPath $txn "/a/b" $context]
		dbxml_error_check_good "txn_commit" [$txn commit] 0

		set txn [start_txn $env]
		set ret1 [$xr next $xv]
		dbxml_error_check_good "txn $op" [$txn $op] 0

		if { $op == "abort" } {
			# If the first 'next' was aborted, another 'next' returns the 
			# original result. 
			set txn [start_txn $env]
			set ret [$xr next $xv]
			dbxml_error_check_good txn_commit [$txn commit] 0
			dbxml_error_check_good next_abort $ret 1
		} else {
			# If first 'next' was committed, we are at the end of 
			# the result set.
			set txn [start_txn $env]
			set ret [$xr next $xv]
			dbxml_error_check_good txn_commit [$txn commit] 0
			dbxml_error_check_good next_commit $ret 0
		}
		delete xv
		delete xr
		delete context
		$container close
	}
	delete container	
}

proc xml003_2_1 { {env "NULL"} {basename $tnum} oargs } {
	puts "\t\tXml003.2.1: Auto_commit not allowed in non-txn env."
	source ./include.tcl
	global errorInfo

	new XmlContainer container $env "$basename.2.1.dbxml" 0

	# Attempt to open with DB_AUTO_COMMIT in non-txn env should fail.
	catch {
	$container open NULL $oargs 0
	} ret

	# Check for correct failure message.
	dbxml_error_check_good auto_commit_without_txn \
	    [is_substr $errorInfo "environment configured for the transaction subsystem"] 1

	delete container	
}

proc xml003_2_2 { {env "NULL"} {basename $tnum} oargs } {
	puts "\t\tXml003.2.2: Auto_commit not allowed within a txn."
	source ./include.tcl

	new XmlContainer container $env "$basename.2.2.dbxml" 0
	
	# Attempt to open with DB_AUTO_COMMIT within explicit txn should fail.
	set txn [start_txn $env] 
	catch {
		$container open $txn $oargs 0
	} ret
	dbxml_error_check_good txn_commit [$txn commit] 0

	# Check for correct failure message.
	dbxml_error_check_good auto_commit_without_txn [is_substr $ret \
	    "may not be specified along with a transaction handle"] 1

	delete container	
}

proc xml003_2_3 { {env "NULL"} {basename $tnum} oargs } {
	puts "\t\tXml003.2.3: Transacted addIndex after auto_commit."
	source ./include.tcl
	xml_cleanup $testdir $env

	# Open container with DB_AUTO_COMMIT flag
	new XmlContainer container $env "$basename.2.3.dbxml" 0
	$container open NULL $oargs 0

	set uri www.uri.com
	set name foo
	set index node-element-presence-none
	# Run addIndex within an explicit transaction.
	set txn [start_txn $env]
	$container addIndex $txn $uri $name $index
	dbxml_error_check_good txn_commit [$txn commit] 0

	$container close
	
	set txn [start_txn $env]
	$container open $txn 0 0
	dbxml_error_check_good txn_commit [$txn commit] 0

	# Test that index spec was persisted.
	wrap XmlIndexSpecification indexes [$container getIndexSpecification NULL]
	set idecl [$indexes find $uri $name]
	delete indexes
	dbxml_error_check_bad foundindex $idecl NULL

	wrap XmlIndexDeclaration xid $idecl
	set retindex [$xid get_index]
	delete xid
	dbxml_error_check_good index $index $retindex
	
	$container close
	delete container	
}

proc xml003_2_4 { {env "NULL"} {basename $tnum} oargs } {
	puts "\t\tXml003.2.4: Transacted putDocument after auto_commit."
	source ./include.tcl
	xml_cleanup $testdir $env

	# Open container with DB_AUTO_COMMIT flag
	new XmlContainer container $env "$basename.2.4.dbxml" 0
	$container open NULL $oargs 0

	new XmlDocument xd
	set data_in "<a><b>c</b></a>"
	$xd setContent $data_in

	# put Document within an explicit transaction
	set txn [start_txn $env]
	set id [$container putDocument $txn $xd]
	dbxml_error_check_good txn_commit [$txn commit] 0

	delete xd

	# Verify that the put was successful by retrieving the data.
	wrap XmlDocument xd [$container getDocument NULL $id]
	set data_out [$xd getContentAsString]
	delete xd
	dbxml_error_check_good content_correct $data_in $data_out
 
	$container close
	delete container
}

proc xml003_2_5 { {env "NULL"} {basename $tnum} oargs } {
	puts "\t\tXml003.2.5: Transacted getDocument after auto_commit."
	source ./include.tcl
	xml_cleanup $testdir $env

	# Open container and putDocument with DB_AUTO_COMMIT flag
	new XmlContainer container $env "$basename.2.5.dbxml" 0
	$container open NULL $oargs 0

	new XmlDocument xd
	set data_in "<a><b>c</b></a>"
	$xd setContent $data_in

	set id [$container putDocument NULL $xd]
	delete xd

	# getDocument within an explicit transaction
	set txn [start_txn $env]
	wrap XmlDocument xd [$container getDocument $txn $id]
	dbxml_error_check_good txn_commit [$txn commit] 0

	# Verify that data is correct
	set data_out [$xd getContentAsString]
	delete xd
	dbxml_error_check_good content_correct $data_in $data_out
 
	$container close
	delete container
}

proc xml003_2_6_1 { {env "NULL"} {basename $tnum} oargs } {
	puts "\t\tXml003.2.6.1: Transacted deleteDocumentByID after auto_commit"
	source ./include.tcl
	xml_cleanup $testdir $env

	# Open container and putDocument with DB_AUTO_COMMIT flag
	new XmlContainer container $env "$basename.2.6.1.dbxml" 0
	$container open NULL $oargs 0

	new XmlDocument xd
	set data_in "<hello/>"
	$xd setContent $data_in

	set id [$container putDocument NULL $xd]
	delete xd

	# deleteDocumentByID within an explicit transaction
	set txn [start_txn $env]
	set ret [$container deleteDocumentByID $txn $id]
	dbxml_error_check_good txn_commit [$txn commit] 0

	# Verify that document has been deleted.
	catch {
		wrap XmlDocument xd [$container getDocument NULL $id]
	} ret 
	dbxml_error_check_good document_deleted $ret "Error: Document not found."
 
	$container close
	delete container
}

proc xml003_2_6_2 { {env "NULL"} {basename $tnum} oargs } {
	puts "\t\tXml003.2.6.2: Transacted delete after auto_commit\
	    \n\t\t\t(using XmlDocument)"
	source ./include.tcl
	xml_cleanup $testdir $env

	# Open container and putDocument with DB_AUTO_COMMIT flag
	new XmlContainer container $env "$basename.2.6.2.dbxml" 0
	$container open NULL $oargs 0

	new XmlDocument xd
	set data_in "<hello/>"
	$xd setContent $data_in

	set id [$container putDocument NULL $xd]

	# delete with XmlDocument within an explicit transaction
	set txn [start_txn $env]
	set ret [$container deleteDocument $txn $xd]
	dbxml_error_check_good txn_commit [$txn commit] 0

	# Verify that document has been deleted.
	catch {
		wrap XmlDocument xd [$container getDocument NULL $id]
	} ret
	dbxml_error_check_good document_deleted $ret "Error: Document not found."

	$container close
	delete container
}

proc xml003_2_7_1 { {env "NULL"} {basename $tnum} oargs } {
	puts "\t\tXml003.2.7.1: Transacted queryWithXPath after\
	    auto_commit \n\t\t\t(using XmlQueryContext)"
	source ./include.tcl
	xml_cleanup $testdir $env

	# Open container and putDocument with DB_AUTO_COMMIT flag
	new XmlContainer container $env "$basename.2.7.1.dbxml" 0
	$container open NULL $oargs 0

	new XmlDocument xd
	new XmlQueryContext context
	set data_in "<a><b>c</b></a>"
	$xd setContent $data_in

	set id [$container putDocument NULL $xd]

	$context setReturnType $XmlQueryContext_ResultDocuments
	$context setEvaluationType $XmlQueryContext_Eager

	# Run transacted query and check return value
	new XmlValue xv
	set txn [start_txn $env]
	wrap XmlResults xr [$container queryWithXPath $txn "/a/b" $context]
	commit_txn $txn
	$xr next $xv

	dbxml_error_check_good "check_contents" [$xv asString] [$xd getContent]
	dbxml_error_check_good "size_of_result_set" [$xr size] 1
	delete xr
	delete xv
	delete xd
	delete context
 
	$container close
	delete container
}


proc xml003_2_7_2 { {env "NULL"} {basename $tnum} oargs } {
	puts "\t\tXml003.2.7.2: Transacted queryWithXPath after\
	    auto_commit \n\t\t\t(using XPathExpression)"
	source ./include.tcl
	xml_cleanup $testdir $env

	# Open container and putDocument with DB_AUTO_COMMIT flag
	new XmlContainer container $env "$basename.2.7.2.dbxml" 0
	$container open NULL $oargs 0

	new XmlDocument xd
	new XmlQueryContext context
	set data_in "<a><b>c</b></a>"
	$xd setContent $data_in

	set id [$container putDocument NULL $xd]

	$context setReturnType $XmlQueryContext_ResultDocuments
	$context setEvaluationType $XmlQueryContext_Eager

	# Run transacted query and check return value
	new XmlValue xv
	set txn [start_txn $env]
	wrap XmlQueryExpression xe [$container parseXPathExpression $txn "/a/b" $context]
	wrap XmlResults xr [$container queryWithXPathExpression $txn $xe]
	commit_txn $txn
	$xr next $xv

	dbxml_error_check_good "check_contents" [$xv asString] [$xd getContent]
	dbxml_error_check_good "size_of_result_set" [$xr size] 1
	delete xe
	delete xr
	delete xv
	delete xd
	delete context
 
	$container close
	delete container
}

proc xml003_2_8 { {env "NULL"} {basename $tnum} oargs } {
	puts "\t\tXml003.2.8: Transacted remove after auto_commit."
	source ./include.tcl
	xml_cleanup $testdir $env

	# Open container and putDocument with DB_AUTO_COMMIT flag
	new XmlContainer container $env "$basename.2.8.dbxml" 0
	$container open NULL $oargs 0

	new XmlDocument xd
	set data_in "<a><b>c</b></a>"
	$xd setContent $data_in

	set id [$container putDocument NULL $xd]
	delete xd

	$container close

	# Now do a transacted remove
	set txn [start_txn $env]
	$container remove $txn
	dbxml_error_check_good "txn_commit" [$txn commit] 0

	# Underlying files should be gone.
	dbxml_error_check_good "original primary_dict exists" \
	    [file exists $testdir/3.2.8.dbxml] 0

	delete container
}

proc xml003_2_9 { {env "NULL"} {basename $tnum} oargs } {
	puts "\t\tXml003.2.9: Transacted rename after auto_commit."
	source ./include.tcl
	xml_cleanup $testdir $env

	# Open container and putDocument with DB_AUTO_COMMIT flag
	new XmlContainer container $env "$basename.2.9.dbxml" 0
	$container open NULL $oargs 0

	new XmlDocument xd
	set data_in "<a><b>c</b></a>"
	$xd setContent $data_in

	set id [$container putDocument NULL $xd]
	delete xd
	
	$container close

	# Now do a transacted rename
	set txn [start_txn $env]
	$container rename $txn "$basename.2.9_renamed.dbxml"
	dbxml_error_check_good "txn_commit" [$txn commit] 0

	# Underlying files should be gone.
	dbxml_error_check_good "original primary_dict exists" \
	    [file exists $testdir/3.2.9.dbxml] 0

	# Renamed files should exist ...
	dbxml_error_check_good "renamed primary_dict exists" \
	    [file exists $testdir/3.2.9_renamed.dbxml] 1

	# ... now retrieve the contents ... 
	$container setName "$basename.2.9_renamed.dbxml"
	$container open NULL "$DB_AUTO_COMMIT" 0 
	catch {
		wrap XmlDocument xd [$container getDocument NULL $id]
		set data_out [$xd getContentAsString]
		delete xd
	} ret

	# ... and verify they match the original loaded data.
	dbxml_error_check_good content_correct $data_in $data_out

	$container close
	delete container
}

proc xml003_3 { {env "NULL"} {basename $tnum} oargs } {
	puts "\tXml003.3: DB-XML in combination with regular Berkeley DB"
	source ./include.tcl
	global dict
	xml_cleanup $testdir $env

	# Open regular BDB database.
	puts "\t\tXml003.3a: Set up BDB database."
	set testfile "$basename.3.db"
	set db [eval {berkdb_open -create -btree -env $env\
	    -auto_commit -mode 0644 $testfile}]
	error_check_good dbopen [is_valid_db $db] TRUE

	# Open DB-XML container.
	puts "\t\tXml003.3b: Set up DB-XML database."
	new XmlContainer container $env "$basename.3.dbxml" 0
	set txn [start_txn $env]
	$container open $txn $oargs 0
	dbxml_error_check_good txn_commit [$txn commit] 0

	# Populate BDB and DBXML databases with records from the same source. 
	set count 1
	set nentries 100
	set did [open $dict]

	puts "\t\tXml003.3c: Put/get on both DB's"
	while { [gets $did str] != -1 && $count < $nentries } {
		# Skip dictionary entries containing apostrophes (these
		# result in badly formed XML) and the null string (DB-XML
		# cannot verify the encoding).
		if { $str == "" || [string first ' $str] != -1 } {
			continue
		}

		set key $str
		new XmlDocument xd
		set data_in "<$str/>"

		$xd setContent $data_in
		
		# Put data in XML and non-XML databases within a txn.
		set txn [start_txn $env]
		set id [$container putDocument $txn $xd]
		set ret [eval {$db put} -txn $txn $key $str]
		dbxml_error_check_good put_commit [$txn commit] 0
		delete xd

		# Get data from both DB's within a txn and make sure it matches.
		set txn [start_txn $env]
		set ret [eval {$db get} -txn $txn {$key}]
		wrap XmlDocument xd [$container getDocument $txn $id]
		set data_out [$xd getContentAsString]
		delete xd
		dbxml_error_check_good get_commit [$txn commit] 0

		# Now check that both values were updated and that they match.
		error_check_good bdb_get $ret [list [list $key $str]]

		dbxml_error_check_good content_correct $data_in $data_out
		dbxml_error_check_good db_matches_dbxml \
			[is_substr $data_out $str] 1

		incr count
	}
	close $did

	error_check_good db_close [$db close] 0
	error_check_good dbxml_close [$container close] ""
	delete container
}


