//
// See the file LICENSE for redistribution information.
//
// Copyright (c) 2002-2003
//	Sleepycat Software.  All rights reserved.
//

//
// XML Path Language (XPath) Version 1.0
// W3C Recommendation 16 November 1999
// http://www.w3.org/TR/1999/REC-xpath-19991116
//

header
{
	#include "dbxml/XmlPortability.hpp"
	#include "dbxml/XmlException.hpp"
	#include "QueryPlan.hpp"
	#include "tokenizer.hpp"
	#include <cmath>

	using namespace antlr;
	using namespace DbXml;
}

options
{
	language="Cpp";
}

// ================================================================================
// T R E E P A R S E R
// ================================================================================

// Transform an AST built from an XPath expression into a physical query plan.
// This tree parser also implements a couple of optimizations.
// 1) Any XPath sub-expressions for which there is no index support
//    are ignored.
// 2) Constant sub-expressions are reduced to their resultant values.
//

{

	// Utility function to count the number of children of an AST node.
	//
	int ASTNumChildren(AST *ast)
	{
		int n= 0;
		RefAST rast= ast->getFirstChild();
		while(rast)
		{
			++n;
			rast= rast->getNextSibling();
		}
		return n;
	}
}

class XPathSelectionTreeParser extends TreeParser;

options
{
    buildAST= false;
	defaultErrorHandler= false; // The default handler sends output to stderr
	importVocab=XPathParser;
}

xpath [XmlQueryContext &context] returns [QueryPlan::SharedPtr r]
  : #(XPATH r=expr[context])
  ;

expr [XmlQueryContext &context] returns [QueryPlan::SharedPtr r]
  {
	QueryPlan::SharedPtr a;
	QueryPlan::SharedPtr b;
	QueryPlan::SharedPtr c;
  }

  : #(CHAIN a=expr[context] b=expr[context])
  {
	// Optimize away constants.
	// JCM - What if both are constants?
	if(!a || QueryPlan::is(a,QueryPlan::CONSTANT))
	{
		r=b;
	}
	else if(!b || QueryPlan::is(b,QueryPlan::CONSTANT))
	{
		r=a;
	}
	else
	{
		r.reset(new PlaceHolderQP(QueryPlan::GLUE,"@",a,b));
	}
  }

  | #(PREDICATE a=expr[context])
  {
	r= a;
  }

  | ROOT
  {
	r.reset(new SequentialScanQP);
	r->setReturnType(QueryPlan::RETURNTYPE_RESULTSET);
  }

  | #(DISCOVER a3:AXIS b3:nodeTest)
  {
	if(#b3!=0)
	{
		// The parser adds a '.any/' prefix to the name when it appears after a '//'.
		//
		// *                     => name='/*'
		// /*                    => name='/*'
		// //*                   => name='.any/*'
		// //foo[bar]            => name='.any/foo',name='bar'
		// //foo[text()='bar']   => name='.any/foo',name='*foo'
		// //*[text()='bar']     => name='.any/*',text (*1)
		//
		// *1 - would need an index that does not have the node type and name
		//      before the value in the key.
		//
		// jcm - /foo/bar/foo/parent::bar/parent::foo
		//
		std::string n(#b3->getText());
		std::string n1;
		std::string n2;
		bool anywhere= false;
		std::string::size_type p1 = n.find("\\");
		if(p1!=std::string::npos) {
			std::string::size_type p2 = n.find("\\", p1+1);
			if(p2!=std::string::npos) {
				anywhere= strncmp(n.c_str(),".any",4)==0;
				n1= n.substr(p1+1, p2-p1-1);
				n2= n.substr(p2+1);
			} else {
				n1= n.substr(0, p1);
				n2= n.substr(p1+1);
				anywhere= n1.compare(".any")==0;
			}
			if(!n1.empty() && (n1[0]=='.' || n1[0]=='*')) n1= "";
		}
		if(n2.empty() || n2.compare("*")==0)
		{
			r.reset(new SequentialScanQP());
			r->setReturnType(QueryPlan::RETURNTYPE_RESULTSET);
		}
		else
		{
			switch(#b3->getType())
			{
			case NAME:
				r.reset(new StepQP(a3->getText(), n1, n2));
				((StepQP*)r.get())->setContextNode(true);
				break;
			case NODE:
				//
				r.reset(new StepQP(a3->getText(), n1, n2));
				((StepQP*)r.get())->setContextNode(true);
				break;
			case TEXT:
				//
				// If the query doesn't explicitly state which node we want the text value
				// of then we get no help from the indexes.
				//
				// Explicit    : //foo[text()='bar']
				// Not Explicit: //*[text()='bar']
				//
				r.reset(new StepQP(a3->getText(), n1, n2));
				((StepQP*)r.get())->setContextNode(true);
				break;
			case COMMENT:
			case PROCESSING_INSTRUCTION:
				//
				// Comment nodes are not indexed
				// Processing Instruction nodes are not indexed
				//
				r.reset(new SequentialScanQP());
				break;
			}
			if(anywhere)
			{
				r->setReturnType(QueryPlan::RETURNTYPE_RESULTSET);
				((StepQP*)r.get())->setAnywhere(anywhere);
			}
		}
	}
  }

  | #(UNION a=expr[context] b=expr[context])
  {
	// Optimize away constants.
	// JCM - What if both are constants?
	if(!a || QueryPlan::is(a,QueryPlan::CONSTANT))
	{
		r= b;
	}
	else if(!b || QueryPlan::is(b,QueryPlan::CONSTANT))
	{
		r= a;
	}
	else
	{
		r.reset(new SetOperationUnionQP(a,b));
	}
  }

  | #(OR a=expr[context] b=expr[context])
  {
	bool isbool_a= false;
	bool isbool_b= false;
	bool b_a= false;
	bool b_b= false;
	ConstantQP *cqp_a= (ConstantQP*)a.get();
	if(QueryPlan::is(a,QueryPlan::CONSTANT))
	{
		isbool_a= true;
		b_a= cqp_a->getValue().asBoolean(&context);
	}
	if(QueryPlan::is(b,QueryPlan::CONSTANT))
	{
		isbool_b= true;
		ConstantQP *cqp_b= (ConstantQP*)b.get();
		b_b= cqp_b->getValue().asBoolean(&context);
	}
	if(isbool_a && isbool_b)
	{
		cqp_a->setBoolean(b_a||b_b);
		r= a;
	}
	else if(isbool_a)
	{
		r= (b_a?a:b);
	}
	else if(isbool_b)
	{
		r= (b_b?b:a);
	}
	else
	{
		r.reset(new PlaceHolderQP(QueryPlan::OR,"|",a,b));
	}
  }

  | #(AND a=expr[context] b=expr[context])
  {
	bool isbool_a= false;
	bool isbool_b= false;
	bool b_a= false;
	bool b_b= false;
	ConstantQP *cqp_a= (ConstantQP*)a.get();
	if(QueryPlan::is(a,QueryPlan::CONSTANT))
	{
		isbool_a= true;
		b_a= cqp_a->getValue().asBoolean(&context);
	}
	if(QueryPlan::is(b,QueryPlan::CONSTANT))
	{
		isbool_b= true;
		ConstantQP *cqp_b= (ConstantQP*)b.get();
		b_b= cqp_b->getValue().asBoolean(&context);
	}
	if(isbool_a && isbool_b)
	{
		cqp_a->setBoolean(b_a&&b_b);
		r= a;
	}
	else if(isbool_a)
	{
		r= (!b_a?a:b);
	}
	else if(isbool_b)
	{
		r= (!b_b?b:a);
	}
	else
	{
		r.reset(new PlaceHolderQP(QueryPlan::AND,"&",a,b));
	}
  }

  | #(EQUAL a=expr[context] b=expr[context])
  {
	if(QueryPlan::is(a,QueryPlan::CONSTANT) &&
	   QueryPlan::is(b,QueryPlan::CONSTANT))
	{
		ConstantQP *cqp_a= (ConstantQP*)a.get();
		ConstantQP *cqp_b= (ConstantQP*)b.get();
		cqp_a->setBoolean(cqp_a->getValue().equals(cqp_b->getValue(),&context));
		r= a;
	}
	else
	{
		if(QueryPlan::is(a,QueryPlan::CONSTANT))
		{
			QueryPlan::SharedPtr t=a; a=b; b=t; // swap(a,b)
		}
		QueryPlan &qp_a= a->rightmost();
		if(qp_a.getType()==QueryPlan::STEP &&
			QueryPlan::is(b,QueryPlan::CONSTANT))
		{
			StepQP *sqp_a= (StepQP*)&qp_a;
			ConstantQP *cqp_b= (ConstantQP*)b.get();
			sqp_a->setOperation(Database::EQUALITY);
			sqp_a->setValue(cqp_b->getValue());
			r= a;
		}
		else
		{
			r.reset(new SequentialScanQP());
		}
	}
  }

  | #(NOTEQUAL a=expr[context] b=expr[context])
  {
	if(QueryPlan::is(a,QueryPlan::CONSTANT) &&
	   QueryPlan::is(b,QueryPlan::CONSTANT))
	{
		ConstantQP *cqp_a= (ConstantQP*)a.get();
		ConstantQP *cqp_b= (ConstantQP*)b.get();
		cqp_a->setBoolean(!cqp_a->getValue().equals(cqp_b->getValue(),&context));
		r= a;
	}
	else
	{
		// There's no index support for inequality.
		r.reset(new SequentialScanQP());
	}
  }

  | #(LTX a=expr[context] b=expr[context])
  {
	// Only supported for Numbers.
	r.reset(0);
	if(QueryPlan::is(a,QueryPlan::CONSTANT) &&
	   QueryPlan::is(b,QueryPlan::CONSTANT))
	{
		ConstantQP *cqp_a= (ConstantQP*)a.get();
		ConstantQP *cqp_b= (ConstantQP*)b.get();
		XmlValue v_a(cqp_a->getValue());
		XmlValue v_b(cqp_b->getValue());
		if(v_a.isNumber(&context) && v_b.isNumber(&context))
		{
			cqp_a->setBoolean(v_a.asNumber(&context) < v_b.asNumber(&context));
			r= a;
		}
	}
	else
	{
		if(QueryPlan::is(a,QueryPlan::CONSTANT))
		{
			QueryPlan::SharedPtr t=a; a=b; b=t; // swap(a,b)
		}
		QueryPlan &qp_a= a->rightmost();
		if(qp_a.getType()==QueryPlan::STEP &&
			QueryPlan::is(b,QueryPlan::CONSTANT))
		{
			StepQP *sqp_a= (StepQP*)&qp_a;
			ConstantQP *cqp_b= (ConstantQP*)b.get();
			XmlValue v_b(cqp_b->getValue());
			if(v_b.isNumber(&context)) // JCM - Should be asNumber
			{
				sqp_a->setOperation(Database::LTX);
				sqp_a->setValue(v_b);
				r= a;
			}
			else
			{
				throw XmlException(XmlException::XPATH_PARSER_ERROR,"Inequality searches (<) are only supported for numbers.");
			}
		}
		else
		{
			r.reset(new SequentialScanQP());
		}
	}
  }

  | #(GTX a=expr[context] b=expr[context])
  {
	// Only supported for Numbers.
	r.reset(0);
	if(QueryPlan::is(a,QueryPlan::CONSTANT) &&
	   QueryPlan::is(b,QueryPlan::CONSTANT))
	{
		ConstantQP *cqp_a= (ConstantQP*)a.get();
		ConstantQP *cqp_b= (ConstantQP*)b.get();
		XmlValue v_a(cqp_a->getValue());
		XmlValue v_b(cqp_b->getValue());
		if(v_a.isNumber(&context) && v_b.isNumber(&context))
		{
			cqp_a->setBoolean(v_a.asNumber(&context) > v_b.asNumber(&context));
			r= a;
		}
	}
	else
	{
		if(QueryPlan::is(a,QueryPlan::CONSTANT))
		{
			QueryPlan::SharedPtr t=a; a=b; b=t; // swap(a,b)
		}
		QueryPlan &qp_a= a->rightmost();
		if(qp_a.getType()==QueryPlan::STEP &&
			QueryPlan::is(b,QueryPlan::CONSTANT))
		{
			StepQP *sqp_a= (StepQP*)&qp_a;
			ConstantQP *cqp_b= (ConstantQP*)b.get();
			XmlValue v_b(cqp_b->getValue());
			if(v_b.isNumber(&context)) // JCM - Should be asNumber
			{
				sqp_a->setOperation(Database::GTX);
				sqp_a->setValue(v_b);
				r= a;
			}
			else
			{
				throw XmlException(XmlException::XPATH_PARSER_ERROR,"Inequality searches (>) are only supported for numbers.");
			}
		}
		else
		{
			r.reset(new SequentialScanQP());
		}
	}
  }

  | #(LTE a=expr[context] b=expr[context])
  {
	// Only supported for Numbers.
	r.reset(0);
	if(QueryPlan::is(a,QueryPlan::CONSTANT) &&
	   QueryPlan::is(b,QueryPlan::CONSTANT))
	{
		ConstantQP *cqp_a= (ConstantQP*)a.get();
		ConstantQP *cqp_b= (ConstantQP*)b.get();
		XmlValue v_a(cqp_a->getValue());
		XmlValue v_b(cqp_b->getValue());
		if(v_a.isNumber(&context) && v_b.isNumber(&context))
		{
			cqp_a->setBoolean(v_a.asNumber(&context) <= v_b.asNumber(&context));
			r= a;
		}
	}
	else
	{
		if(QueryPlan::is(a,QueryPlan::CONSTANT))
		{
			QueryPlan::SharedPtr t=a; a=b; b=t; // swap(a,b)
		}
		QueryPlan &qp_a= a->rightmost();
		if(qp_a.getType()==QueryPlan::STEP &&
			QueryPlan::is(b,QueryPlan::CONSTANT))
		{
			StepQP *sqp_a= (StepQP*)&qp_a;
			ConstantQP *cqp_b= (ConstantQP*)b.get();
			XmlValue v_b(cqp_b->getValue());
			if(v_b.isNumber(&context)) // JCM - Should be asNumber
			{
				sqp_a->setOperation(Database::LTE);
				sqp_a->setValue(v_b);
				r= a;
			}
			else
			{
				throw XmlException(XmlException::XPATH_PARSER_ERROR,"Inequality searches (<=) are only supported for numbers.");
			}
		}
		else
		{
			r.reset(new SequentialScanQP());
		}
	}
  }

  | #(GTE a=expr[context] b=expr[context])
  {
	// Only supported for Numbers.
	r.reset(0);
	if(QueryPlan::is(a,QueryPlan::CONSTANT) &&
	   QueryPlan::is(b,QueryPlan::CONSTANT))
	{
		ConstantQP *cqp_a= (ConstantQP*)a.get();
		ConstantQP *cqp_b= (ConstantQP*)b.get();
		XmlValue v_a(cqp_a->getValue());
		XmlValue v_b(cqp_b->getValue());
		if(v_a.isNumber(&context) && v_b.isNumber(&context))
		{
			cqp_a->setBoolean(v_a.asNumber(&context) >= v_b.asNumber(&context));
			r= a;
		}
	}
	else
	{
		if(QueryPlan::is(a,QueryPlan::CONSTANT))
		{
			QueryPlan::SharedPtr t=a; a=b; b=t; // swap(a,b)
		}
		QueryPlan &qp_a= a->rightmost();
		if(qp_a.getType()==QueryPlan::STEP &&
			QueryPlan::is(b,QueryPlan::CONSTANT))
		{
			StepQP *sqp_a= (StepQP*)&qp_a;
			ConstantQP *cqp_b= (ConstantQP*)b.get();
			XmlValue v_b(cqp_b->getValue());
			if(v_b.isNumber(&context)) // JCM - Should be asNumber
			{
				sqp_a->setOperation(Database::GTE);
				sqp_a->setValue(v_b);
				r= a;
			}
			else
			{
				throw XmlException(XmlException::XPATH_PARSER_ERROR,"Inequality searches (>=) are only supported for numbers.");
			}
		}
		else
		{
			r.reset(new SequentialScanQP());
		}
	}
  }

  | #(PLUS a=expr[context] b=expr[context])
  {
	r.reset(0);
	if(QueryPlan::is(a,QueryPlan::CONSTANT) &&
	   QueryPlan::is(b,QueryPlan::CONSTANT))
	{
		ConstantQP *cqp_a= (ConstantQP*)a.get();
		ConstantQP *cqp_b= (ConstantQP*)b.get();
		XmlValue v_a(cqp_a->getValue());
		XmlValue v_b(cqp_b->getValue());
		if(v_a.isNumber(&context) && v_b.isNumber(&context))
		{
			cqp_a->setNumber(v_a.asNumber(&context) + v_b.asNumber(&context));
			r= a;
		}
	}
	else
	{
		r.reset(new SequentialScanQP());
	}
  }

  | (#(MINUS expr[context] expr[context]))=>#(MINUS a=expr[context] b=expr[context])
  {
	r.reset(0);
	if(QueryPlan::is(a,QueryPlan::CONSTANT) &&
	   QueryPlan::is(b,QueryPlan::CONSTANT))
	{
		ConstantQP *cqp_a= (ConstantQP*)a.get();
		ConstantQP *cqp_b= (ConstantQP*)b.get();
		XmlValue v_a(cqp_a->getValue());
		XmlValue v_b(cqp_b->getValue());
		if(v_a.isNumber(&context) && v_b.isNumber(&context))
		{
			cqp_a->setNumber(v_a.asNumber(&context) - v_b.asNumber(&context));
			r= a;
		}
	}
	else
	{
		r.reset(new SequentialScanQP());
	}
  }

  | #(MINUS a=expr[context])
  {
	r.reset(0);
	if(QueryPlan::is(a,QueryPlan::CONSTANT))
	{
		ConstantQP *cqp_a= (ConstantQP*)a.get();
		XmlValue v_a(cqp_a->getValue());
		if(v_a.isNumber(&context))
		{
			cqp_a->setNumber(-v_a.asNumber(&context));
			r= a;
		}
		else
		{
		 	throw XmlException(XmlException::XPATH_PARSER_ERROR,"Minus Literal (eg. -123) is only supported for numbers.");
		}
	}
	else
	{
		r.reset(new SequentialScanQP());
	}
  }

  | #(STAR a=expr[context] b=expr[context])
  {
	r.reset(0);
	if(QueryPlan::is(a,QueryPlan::CONSTANT) &&
	   QueryPlan::is(b,QueryPlan::CONSTANT))
	{
		ConstantQP *cqp_a= (ConstantQP*)a.get();
		ConstantQP *cqp_b= (ConstantQP*)b.get();
		XmlValue v_a(cqp_a->getValue());
		XmlValue v_b(cqp_b->getValue());
		if(v_a.isNumber(&context) && v_b.isNumber(&context))
		{
			cqp_a->setNumber(v_a.asNumber(&context) * v_b.asNumber(&context));
			r= a;
		}
	}
	else
	{
		r.reset(new SequentialScanQP());
	}
  }

  | #(DIV a=expr[context] b=expr[context])
  {
	r.reset(0);
	if(QueryPlan::is(a,QueryPlan::CONSTANT) &&
	   QueryPlan::is(b,QueryPlan::CONSTANT))
	{
		ConstantQP *cqp_a= (ConstantQP*)a.get();
		ConstantQP *cqp_b= (ConstantQP*)b.get();
		XmlValue v_a(cqp_a->getValue());
		XmlValue v_b(cqp_b->getValue());
		if(v_a.isNumber(&context) && v_b.isNumber(&context))
		{
			cqp_a->setNumber(v_a.asNumber(&context) / v_b.asNumber(&context));
			r= a;
		}
	}
	else
	{
		r.reset(new SequentialScanQP());
	}
  }

  | #(MOD a=expr[context] b=expr[context])
  {
	r.reset(0);
	if(QueryPlan::is(a,QueryPlan::CONSTANT) &&
	   QueryPlan::is(b,QueryPlan::CONSTANT))
	{
		ConstantQP *cqp_a= (ConstantQP*)a.get();
		ConstantQP *cqp_b= (ConstantQP*)b.get();
		XmlValue v_a(cqp_a->getValue());
		XmlValue v_b(cqp_b->getValue());
		if(v_a.isNumber(&context) && v_b.isNumber(&context))
		{
			cqp_a->setNumber(fmod(v_a.asNumber(&context),v_b.asNumber(&context)));
			r= a;
		}
	}
	else
	{
		r.reset(new SequentialScanQP());
	}
  }

  | a19:VARIABLE
  {
	XmlValue v(XmlValue::VARIABLE,a19->getText());
    r.reset(new ConstantQP(v));
  }

  | a20:LITERAL
  { 
    std::string s(a20->getText());
	s= s.substr(1,s.length()-2); // Strip off quotes.
	XmlValue v(s);
    r.reset(new ConstantQP(v));
  }

  | a21:NUMBER
  {
	XmlValue v(atof(a21->getText().c_str()));
	r.reset(new ConstantQP(v));
  }

  | a22:FUNCTION
  {
   	r.reset(0);
	std::string fn= a22->getText();
	int na= ASTNumChildren(a22);
	// last
	// position
	// count
	if(fn.compare("id")==0 && na==1)
	{
		RefAST arg1= a22->getFirstChild();
		a= expr(arg1,context);
		if(QueryPlan::is(a,QueryPlan::CONSTANT))
		{
			ConstantQP *cqp_a= (ConstantQP*)a.get();
			std::string s(cqp_a->getValue().asString(&context));
			tokenizer tok(s);
			for(string val; tok.next(val) == 0; )
			{
				QueryPlan::SharedPtr sqp(new StepQP("attribute","id",""));
				((StepQP *)sqp.get())->setValue(val);
				((StepQP *)sqp.get())->setOperation(Database::EQUALITY);
				sqp->setReturnType(QueryPlan::RETURNTYPE_RESULTSET);
				if(!r)
				{
					r= sqp;
				}
				else
				{
					r.reset(new PlaceHolderQP(QueryPlan::OR,"|",sqp,r));
				}
			}
		}
	}
	// local-name
	// namespace-uri
	else if(fn.compare("name")==0 && na==0)
	{
		// JCM - Implement support for '[name()='foo']'
		//
		// A step must come before the predicate, and the axis will tell
		// us what kind of node we're dealing with... 
		//
		// '/*[name()='x']' is the same as '/x'
		// '/*[name()=$x]' is interesting
	}
	else if(fn.compare("starts-with")==0 && na==2)
	{
		RefAST arg1= a22->getFirstChild();
		RefAST arg2= arg1->getNextSibling();
		a= expr(arg1,context);
		b= expr(arg2,context);
		if(QueryPlan::is(a,QueryPlan::CONSTANT) &&
		   QueryPlan::is(b,QueryPlan::CONSTANT))
		{
			ConstantQP *cqp_a= (ConstantQP*)a.get();
			ConstantQP *cqp_b= (ConstantQP*)b.get();
			cqp_a->setValue(XmlValue::startsWith(&context,cqp_a->getValue(),cqp_b->getValue()));
			r= a;
		}
		else
		{
			QueryPlan &qp_a= a->rightmost();
			if(qp_a.getType()==QueryPlan::STEP &&
			   QueryPlan::is(b,QueryPlan::CONSTANT))
			{
				StepQP *sqp_a= (StepQP*)&qp_a;
				ConstantQP *cqp_b= (ConstantQP*)b.get();
				sqp_a->setOperation(Database::PREFIX);
				sqp_a->setValue(cqp_b->getValue());
				r= a;
			}
		}
	}
	else if(fn.compare("contains")==0 && na==2)
	{
		RefAST arg1= a22->getFirstChild();
		RefAST arg2= arg1->getNextSibling();
		a= expr(arg1,context);
		b= expr(arg2,context);
		if(QueryPlan::is(a,QueryPlan::CONSTANT) &&
		   QueryPlan::is(b,QueryPlan::CONSTANT))
		{
			ConstantQP *cqp_a= (ConstantQP*)a.get();
			ConstantQP *cqp_b= (ConstantQP*)b.get();
			cqp_a->setValue(XmlValue::contains(&context,cqp_a->getValue(),cqp_b->getValue()));
			r= a;
		}
		else
		{
			QueryPlan &qp_a= a->rightmost();
			if(qp_a.getType()==QueryPlan::STEP &&
			   QueryPlan::is(b,QueryPlan::CONSTANT))
			{
				StepQP *sqp_a= (StepQP*)&qp_a;
				ConstantQP *cqp_b= (ConstantQP*)b.get();
				sqp_a->setOperation(Database::SUBSTRING);
				sqp_a->setValue(cqp_b->getValue());
				r= a;
			}
		}
	}
	else if(fn.compare("string")==0 && na==1)
	{
		RefAST arg1= a22->getFirstChild();
		a= expr(arg1,context);
		if(QueryPlan::is(a,QueryPlan::CONSTANT))
		{
			ConstantQP *cqp_a= (ConstantQP*)a.get();
			cqp_a->setValue(XmlValue::string(&context,cqp_a->getValue()));
			r= a;
		}
	}
	else if(fn.compare("concat")==0 && na>=2)
	{
		string s;
		bool failed= false;
		RefAST rast= a22->getFirstChild();
		while(rast && !failed)
		{
			a= expr(rast,context);
			if(QueryPlan::is(a,QueryPlan::CONSTANT))
			{
				ConstantQP *cqp_a= (ConstantQP*)a.get();
				s+= cqp_a->getValue().asString(&context);
				if(!r)
				{
					r= a;
				}
			}
			else
			{
				failed= true;
			}
			rast= rast->getNextSibling();
		}
		if(!failed)
		{
			((ConstantQP*)r.get())->setString(s);
		}
	}
	else if(fn.compare("substring-before")==0 && na==2)
	{
		RefAST arg1= a22->getFirstChild();
		RefAST arg2= arg1->getNextSibling();
		a= expr(arg1,context);
		b= expr(arg2,context);
		if(QueryPlan::is(a,QueryPlan::CONSTANT) &&
		   QueryPlan::is(b,QueryPlan::CONSTANT))
		{
			ConstantQP *cqp_a= (ConstantQP*)a.get();
			ConstantQP *cqp_b= (ConstantQP*)b.get();
			cqp_a->setValue(XmlValue::substringBefore(&context,cqp_a->getValue(),cqp_b->getValue()));
			r= a;
		}
		else
		{
			QueryPlan &qp_a= a->rightmost();
			if(qp_a.getType()==QueryPlan::STEP &&
			   QueryPlan::is(b,QueryPlan::CONSTANT))
			{
				StepQP *sqp_a= (StepQP*)&qp_a;
				ConstantQP *cqp_b= (ConstantQP*)b.get();
				sqp_a->setOperation(Database::SUBSTRING);
				sqp_a->setValue(cqp_b->getValue());
				r= a;
			}
		}
	}
	else if(fn.compare("substring-after")==0 && na==2)
	{
		RefAST arg1= a22->getFirstChild();
		RefAST arg2= arg1->getNextSibling();
		a= expr(arg1,context);
		b= expr(arg2,context);
		if(QueryPlan::is(a,QueryPlan::CONSTANT) &&
		   QueryPlan::is(b,QueryPlan::CONSTANT))
		{
			ConstantQP *cqp_a= (ConstantQP*)a.get();
			ConstantQP *cqp_b= (ConstantQP*)b.get();
			cqp_a->setValue(XmlValue::substringAfter(&context,cqp_a->getValue(),cqp_b->getValue()));
			r= a;
		}
		else
		{
			QueryPlan &qp_a= a->rightmost();
			if(qp_a.getType()==QueryPlan::STEP &&
			   QueryPlan::is(b,QueryPlan::CONSTANT))
			{
				StepQP *sqp_a= (StepQP*)&qp_a;
				ConstantQP *cqp_b= (ConstantQP*)b.get();
				sqp_a->setOperation(Database::SUBSTRING);
				sqp_a->setValue(cqp_b->getValue());
				r= a;
			}
		}
	}
	else if(fn.compare("substring")==0 && (na==2 || na==3))
	{
		RefAST arg1= a22->getFirstChild();
		RefAST arg2= arg1->getNextSibling();
		a= expr(arg1,context);
		b= expr(arg2,context);
		if(QueryPlan::is(a,QueryPlan::CONSTANT) &&
		   QueryPlan::is(b,QueryPlan::CONSTANT))
		{
			ConstantQP *cqp_a= (ConstantQP*)a.get();
			ConstantQP *cqp_b= (ConstantQP*)b.get();
			if(na==2)
			{
				cqp_a->setValue(XmlValue::substring(&context,cqp_a->getValue(),cqp_b->getValue()));
				r= a;
			}
			else
			{
				RefAST arg3= arg2->getNextSibling();
				c= expr(arg3,context);
				if(QueryPlan::is(c,QueryPlan::CONSTANT))
				{
					ConstantQP *cqp_c= (ConstantQP*)c.get();
					cqp_a->setValue(XmlValue::substring(&context,cqp_a->getValue(),cqp_b->getValue(),cqp_c->getValue()));
					r= a;
				}
			}
		}
	}
	else if(fn.compare("string-length")==0 && na==1)
	{
		RefAST arg1= a22->getFirstChild();
		a= expr(arg1,context);
		if(QueryPlan::is(a,QueryPlan::CONSTANT))
		{
			ConstantQP *cqp_a= (ConstantQP*)a.get();
			cqp_a->setValue(XmlValue::stringLength(&context,cqp_a->getValue()));
			r= a;
		}
	}
	else if(fn.compare("normalize-space")==0 && na==1)
	{
		RefAST arg1= a22->getFirstChild();
		a= expr(arg1,context);
		if(QueryPlan::is(a,QueryPlan::CONSTANT))
		{
			ConstantQP *cqp_a= (ConstantQP*)a.get();
			cqp_a->setValue(XmlValue::normalizeSpace(&context,cqp_a->getValue()));
			r= a;
		}
	}
	else if(fn.compare("translate")==0 && na==3)
	{
		RefAST arg1= a22->getFirstChild();
		RefAST arg2= arg1->getNextSibling();
		RefAST arg3= arg2->getNextSibling();
		a= expr(arg1,context);
		b= expr(arg2,context);
		c= expr(arg3,context);
		if(QueryPlan::is(a,QueryPlan::CONSTANT) &&
		   QueryPlan::is(b,QueryPlan::CONSTANT) &&
		   QueryPlan::is(c,QueryPlan::CONSTANT))
		{
			ConstantQP *cqp_a= (ConstantQP*)a.get();
			ConstantQP *cqp_b= (ConstantQP*)b.get();
			ConstantQP *cqp_c= (ConstantQP*)c.get();
			cqp_a->setValue(XmlValue::translate(&context,cqp_a->getValue(),cqp_b->getValue(),cqp_c->getValue()));
			r= a;
		}
	}
	else if(fn.compare("boolean")==0 && na==1)
	{
		RefAST arg1= a22->getFirstChild();
		a= expr(arg1,context);
		if(QueryPlan::is(a,QueryPlan::CONSTANT))
		{
			ConstantQP *cqp_a= (ConstantQP*)a.get();
			cqp_a->setValue(XmlValue::boolean(&context,cqp_a->getValue()));
			r= a;
		}
	}
	else if(fn.compare("not")==0 && na==1)
	{
		RefAST arg1= a22->getFirstChild();
		a= expr(arg1,context);
		if(QueryPlan::is(a,QueryPlan::CONSTANT))
		{
			ConstantQP *cqp_a= (ConstantQP*)a.get();
			cqp_a->setValue(XmlValue::booleanNot(&context,cqp_a->getValue()));
			r= a;
		}
		else
		{
			// The query plan for the contained expression is discarded.
			// This is OK, as predicates like [not(x='y')] will map onto
			// a SequentialScan, which is OK. Double negative expressions
			// like [not(x!='y')] will not be mapped onto anything clever
			// though... perhaps an exercise for you dear reader.
		}
	}
	else if(fn.compare("true")==0 && na==0)
	{
		XmlValue v(XmlValue::booleanTrue(&context));
		r.reset(new ConstantQP(v));
	}
	else if(fn.compare("false")==0 && na==0)
	{
		XmlValue v(XmlValue::booleanFalse(&context));
		r.reset(new ConstantQP(v));
	}
	else if(fn.compare("number")==0 && na==1)
	{
		RefAST arg1= a22->getFirstChild();
		a= expr(arg1,context);
		if(QueryPlan::is(a,QueryPlan::CONSTANT))
		{
			ConstantQP *cqp_a= (ConstantQP*)a.get();
			cqp_a->setValue(XmlValue::number(&context,cqp_a->getValue()));
			r= a;
		}
	}
	else if(fn.compare("floor")==0 && na==1)
	{
		RefAST arg1= a22->getFirstChild();
		a= expr(arg1,context);
		if(QueryPlan::is(a,QueryPlan::CONSTANT))
		{
			ConstantQP *cqp_a= (ConstantQP*)a.get();
			XmlValue v_a(cqp_a->getValue());
			if(v_a.isNumber(&context))
			{
				cqp_a->setValue(XmlValue::floor(&context,v_a));
				r= a;
			}
		}
	}
	else if(fn.compare("ceiling")==0 && na==1)
	{
		RefAST arg1= a22->getFirstChild();
		a= expr(arg1,context);
		if(QueryPlan::is(a,QueryPlan::CONSTANT))
		{
			ConstantQP *cqp_a= (ConstantQP*)a.get();
			XmlValue v_a(cqp_a->getValue());
			if(v_a.isNumber(&context))
			{
				cqp_a->setValue(XmlValue::ceiling(&context,v_a));
				r= a;
			}
		}
	}
	else if(fn.compare("round")==0 && na==1)
	{
		RefAST arg1= a22->getFirstChild();
		a= expr(arg1,context);
		if(QueryPlan::is(a,QueryPlan::CONSTANT))
		{
			ConstantQP *cqp_a= (ConstantQP*)a.get();
			XmlValue v_a(cqp_a->getValue());
			cqp_a->setValue(XmlValue::round(&context,v_a));
			r= a;
		}
	}
	else
	{
		// Indexes don't support this function...
	}
	if(!r)
	{
		r.reset(new SequentialScanQP());
	}
  }
  ;

nodeTest: NAME | COMMENT | TEXT | NODE | PROCESSING_INSTRUCTION;
