%module dbxml_java

%include "various.i"

%javaconst(0);

%pragma(java) jniclasscode=%{
  static {
    // An alternate library name can be specified via a property.
    //
    String override;

    if ((override = System.getProperty("sleepycat.dbxml.libfile")) != null) {
      System.load(override);
    }
    else if ((override = System.getProperty("sleepycat.dbxml.libname")) != null) {
      System.loadLibrary(override);
    }
    else {
      String os = System.getProperty("os.name");
      if (os != null && os.startsWith("Windows")) {
        // library name is "libdbxml_java10.dll" (for example) on Windows
	String libname = "libdbxml_java" +
            XmlConstants.DBXML_VERSION_MAJOR + XmlConstants.DBXML_VERSION_MINOR;
        // add trailing "d" for debug build
	if (XmlConstants.DBXML_DEBUG)
		libname += "d";
        System.loadLibrary(libname);
      }
      else {
        // library name is "libdbxml_java-1.0.so" (for example) on UNIX
        // Note: "dbxml_java" isn't good enough;
        // some Unixes require us to use the explicit SONAME.
        System.loadLibrary("dbxml_java-" +
                   XmlConstants.DBXML_VERSION_MAJOR + "." +
                   XmlConstants.DBXML_VERSION_MINOR);
      }
    }

    initialize();
  }

  static native final void initialize();
%}

%typemap(check) SWIGTYPE *self %{
	if (!$input) {
		jenv->Throw((jthrowable)jenv->NewObject(xmlex_class, xmlex_construct,
		    XmlException::INTERNAL_ERROR,
		    jenv->NewStringUTF("null object - call after object destroyed?"),
		    EINVAL));
		return $null;
	}
%}

%typemap(check) SWIGTYPE *jcls %{
	UNUSED($input);	/* just in case */
%}

/* Move global constants into the XmlContainer class */
%typemap(javacode) XmlContainer %{
  public final static int LEVEL_NONE = dbxml_java.LEVEL_NONE;
  public final static int LEVEL_DEBUG = dbxml_java.LEVEL_DEBUG;
  public final static int LEVEL_INFO = dbxml_java.LEVEL_INFO;
  public final static int LEVEL_WARNING = dbxml_java.LEVEL_WARNING;
  public final static int LEVEL_ERROR = dbxml_java.LEVEL_ERROR;
  public final static int LEVEL_ALL = dbxml_java.LEVEL_ALL;
  public final static int CATEGORY_NONE = dbxml_java.CATEGORY_NONE;
  public final static int CATEGORY_INDEXER = dbxml_java.CATEGORY_INDEXER;
  public final static int CATEGORY_QUERY = dbxml_java.CATEGORY_QUERY;
  public final static int CATEGORY_OPTIMIZER = dbxml_java.CATEGORY_OPTIMIZER;
  public final static int CATEGORY_DICTIONARY = dbxml_java.CATEGORY_DICTIONARY;
  public final static int CATEGORY_CONTAINER = dbxml_java.CATEGORY_CONTAINER;
  public final static int CATEGORY_ALL = dbxml_java.CATEGORY_ALL;
  public final static String metaDataNamespace_uri = dbxml_java.metaDataNamespace_uri;
  public final static String metaDataNamespace_prefix = dbxml_java.metaDataNamespace_prefix;
  public final static String metaDataName_id = dbxml_java.metaDataName_id;
  public final static String metaDataName_name = dbxml_java.metaDataName_name;
  public final static String metaDataName_content = dbxml_java.metaDataName_content;
  public final static String metaDataName_default = dbxml_java.metaDataName_default;
  public final static String metaDataNamespace_prefix_debug = dbxml_java.metaDataNamespace_prefix_debug;
%}

%{
/* don't use SWIG's array handling - save code space */
#define SWIG_NOINCLUDE 1

jclass dbt_class;
jclass env_class;
jclass txn_class;
jclass xml_indexdecl_class;

jclass xmlex_class;
jclass dbex_class;

jfieldID dbt_data_fid;
jfieldID dbt_size_fid;
jfieldID dbt_ulen_fid;
jfieldID dbt_dlen_fid;
jfieldID dbt_doff_fid;

jfieldID env_private_fid;
jfieldID txn_private_fid;

jmethodID dbt_construct;
jmethodID xml_indexdecl_construct;
jmethodID xmlex_construct;
jmethodID dbex_construct;

extern "C"
JNIEXPORT void JNICALL Java_com_sleepycat_dbxml_dbxml_1javaJNI_initialize
  (JNIEnv *jenv, jclass clazz)
{
	(void)clazz; /* unused */
	
#define DB_PKG "com/sleepycat/db/"
#define XML_PKG "com/sleepycat/dbxml/"

	/* XXX TODO: better error handling during initialization */
	dbt_class = jenv->FindClass(DB_PKG "Dbt");
	env_class = jenv->FindClass(DB_PKG "DbEnv");
	txn_class = jenv->FindClass(DB_PKG "DbTxn");
	xml_indexdecl_class = jenv->FindClass(XML_PKG "XmlIndexDeclaration");
	
	xmlex_class = jenv->FindClass("com/sleepycat/dbxml/XmlException");
	dbex_class = jenv->FindClass(DB_PKG "DbException");

	if (!env_class || !txn_class || !xmlex_class || !dbex_class) {
		fprintf(stderr, "Failed to load com.sleepycat.db.* - check CLASSPATH\n");
		return;
	}

	/* Wrap classes in GlobalRefs so that we can keep them between calls */
	dbt_class = (jclass)jenv->NewGlobalRef(dbt_class);
	env_class = (jclass)jenv->NewGlobalRef(env_class);
	txn_class = (jclass)jenv->NewGlobalRef(txn_class);
	xml_indexdecl_class = (jclass)jenv->NewGlobalRef(xml_indexdecl_class);

	xmlex_class = (jclass)jenv->NewGlobalRef(xmlex_class);
	dbex_class = (jclass)jenv->NewGlobalRef(dbex_class);

	/* Get field IDs */
	dbt_data_fid = jenv->GetFieldID(dbt_class, "data", "[B");
	dbt_size_fid = jenv->GetFieldID(dbt_class, "size", "I");
	dbt_ulen_fid = jenv->GetFieldID(dbt_class, "ulen", "I");
	dbt_dlen_fid = jenv->GetFieldID(dbt_class, "dlen", "I");
	dbt_doff_fid = jenv->GetFieldID(dbt_class, "doff", "I");

#if DB_VERSION_MAJOR < 4 || (DB_VERSION_MAJOR == 4 && DB_VERSION_MINOR <= 1)
	env_private_fid = jenv->GetFieldID(env_class, "private_dbobj_", "J");
	txn_private_fid = jenv->GetFieldID(txn_class, "private_dbobj_", "J");
#else
	env_private_fid = jenv->GetFieldID(env_class, "swigCPtr", "J");
	txn_private_fid = jenv->GetFieldID(txn_class, "swigCPtr", "J");
#endif

	/* Get method IDs */
	dbt_construct = jenv->GetMethodID(dbt_class, "<init>",
	    "()V");
	xml_indexdecl_construct = jenv->GetMethodID(xml_indexdecl_class, "<init>",
	    "(Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;)V");

	xmlex_construct = jenv->GetMethodID(xmlex_class, "<init>",
	    "(ILjava/lang/String;I)V");
	dbex_construct = jenv->GetMethodID(dbex_class, "<init>",
	    "(Ljava/lang/String;I)V");
}

/* Union to convert longs to pointers (see {get,set}_private_dbobj). */
typedef union {
	jlong java_long;
	void *ptr;
} long_to_ptr;

static DbEnv *get_DbEnv(JNIEnv *jenv, jobject jdbenv)
{
	if (jdbenv == NULL)
		return NULL;
	else {
#if DB_VERSION_MAJOR < 4 || (DB_VERSION_MAJOR == 4 && DB_VERSION_MINOR <= 1)
		long_to_ptr lp;
		lp.java_long = jenv->GetLongField(jdbenv, env_private_fid);
		return DbEnv::wrap_DB_ENV((DB_ENV *)lp.ptr);
#else
		jlong swigCPtr = jenv->GetLongField(jdbenv, env_private_fid);
		return DbEnv::wrap_DB_ENV(*(DB_ENV **)&swigCPtr);
#endif
	}
}

static DbTxn *get_DbTxn(JNIEnv *jenv, jobject jtxn)
{
	if (jtxn == NULL)
		return NULL;
	else {
#if DB_VERSION_MAJOR < 4 || (DB_VERSION_MAJOR == 4 && DB_VERSION_MINOR <= 1)
		long_to_ptr lp;
		lp.java_long = jenv->GetLongField(jtxn, txn_private_fid);
		return DbTxn::wrap_DB_TXN((DB_TXN *)lp.ptr);
#else
		jlong swigCPtr = jenv->GetLongField(jtxn, txn_private_fid);
		return DbTxn::wrap_DB_TXN(*(DB_TXN **)&swigCPtr);
#endif
	}
}

struct DBT_INFO {
	Dbt dbt;
	jbyte *orig_data;
};

%}

%exception {
	jthrowable t = NULL;

	try {
		$action
	} catch (XmlException &xe) {
		t = (jthrowable)jenv->NewObject(xmlex_class, xmlex_construct,
		    xe.getExceptionCode(), jenv->NewStringUTF(xe.what()),
		    xe.getDbErrno());
	} catch (...) {
		fprintf(stderr, "Uncaught exception from C++ API!\n");
		t = (jthrowable)jenv->NewObject(dbex_class, dbex_construct,
		    jenv->NewStringUTF("Uncaught exception from C++ API"), 0);
	}

	if (t)
		jenv->Throw(t);
}

// Typemaps
%define JAVA_TYPEMAP(_ctype, _jtype, _jnitype)
%typemap(jstype) _ctype #_jtype
%typemap(jtype) _ctype #_jtype
%typemap(jni) _ctype #_jnitype
%typemap(javain) _ctype "$javainput"
%typemap(javaout) _ctype { return $jnicall; }
%enddef

JAVA_TYPEMAP(u_int32_t, int, jint)
JAVA_TYPEMAP(size_t, int, jint)

%typemap(in) bool %{
	$1 = ($input != JNI_FALSE);
%}

%typemap(out) bool %{
	$result = $1 ? JNI_TRUE : JNI_FALSE;
%}


JAVA_TYPEMAP(DbEnv *, com.sleepycat.db.DbEnv, jobject)
%typemap(in) DbEnv * {
    $1 = get_DbEnv(jenv, $input);
}

JAVA_TYPEMAP(DbTxn *, com.sleepycat.db.DbTxn, jobject)
%typemap(in) DbTxn * {
    $1 = get_DbTxn(jenv, $input);
}

/*
 * Fake DBT * translation to work around SWIG problems with (void *, size_t)
 * typemaps for Java.
 */
JAVA_TYPEMAP(Dbt &, byte[], jbyteArray)
JAVA_TYPEMAP(Dbt *, byte[], jbyteArray)

%typemap(in) Dbt & (struct DBT_INFO ldbt) %{
    if ($input != NULL) {
        ldbt.dbt.set_data(ldbt.orig_data = jenv->GetByteArrayElements($input, (jboolean *)0));
        ldbt.dbt.set_size(jenv->GetArrayLength($input));
    }
    ldbt.dbt.set_flags(DB_DBT_MALLOC);
    $1 = &ldbt.dbt;
%}

%typemap(freearg) Dbt & %{
    if ($input != NULL)
        jenv->ReleaseByteArrayElements($input, ldbt$argnum.orig_data, 0);
%}

%typemap(out) const Dbt * %{
    $result = jenv->NewByteArray($1->get_size());
    jenv->SetByteArrayRegion($result, 0, $1->get_size(), (jbyte *)$1->get_data());
%}


JAVA_TYPEMAP(XmlIndexDeclaration *, XmlIndexDeclaration, jobject)
%typemap(out) XmlIndexDeclaration * %{
    if ($1 == NULL)
        $result = NULL;
    else {
        $result = jenv->NewObject(xml_indexdecl_class, xml_indexdecl_construct,
            jenv->NewStringUTF($1->uri.c_str()),
            jenv->NewStringUTF($1->name.c_str()),
            jenv->NewStringUTF($1->index.c_str()));
        delete $1;
    }
%}
