%include "exception.i"
%include "typemaps.i"

%{
#include "db_cxx.h"
#include "dbxml/DbXml.hpp"
#include <errno.h>
#include <fstream>

using namespace DbXml;

class XmlIndexDeclaration {
public:
	XmlIndexDeclaration() {}
	XmlIndexDeclaration(const std::string &uri, const std::string &name, const std::string &index) : uri(uri), name(name), index(index) {}

	std::string uri, name, index;
};
%}

#if defined(SWIGJAVA)
%include "std_string.i"
%include "dbxml_java.i"
#elif defined(SWIGPYTHON)
%include "std_string.i"
%include "dbxml_python.i"
#elif defined(SWIGTCL8)
%include "std_string.i"
%include "dbxml_tcl.i"
#elif defined(SWIGCSHARP)
%include "dbxml_csharp.i"
#elif defined(SWIGPHP4)
%include "std_string.i"
%include "dbxml_php4.i"
#else
#error "Unknown SWIG target language"
#endif

#if defined(SWIGJAVA) || defined(SWIGPYTHON) || defined(SWIGCSHARP) || defined(SWIGTCL8)
#define DBXML_USEOVERLOADS
#endif

typedef unsigned int u_int32_t;
typedef int int32_t;

class Dbt;

class XmlDocument;
class XmlContainer;
class XmlIndexSpecification;
class XmlQueryContext;
class XmlResults;
class XmlUpdateContext;
class XmlValue;
class XmlQueryExpression;
class XmlModify;

#ifndef SWIGJAVA
enum DbConstants {
	DB_AUTO_COMMIT, DB_CREATE, DB_DIRTY_READ, DB_EXCL,
	DB_NOMMAP, DB_RDONLY, DB_THREAD
};
#endif

%constant const char *metaDataNamespace_uri = DbXml::metaDataNamespace_uri;
%constant const char *metaDataNamespace_prefix = DbXml::metaDataNamespace_prefix;
%constant const char *metaDataName_id = DbXml::metaDataName_id;
%constant const char *metaDataName_name = DbXml::metaDataName_name;
%constant const char *metaDataName_content = DbXml::metaDataName_content;
%constant const char *metaDataName_default = DbXml::metaDataName_default;
%constant const char *metaDataNamespace_prefix_debug = DbXml::metaDataNamespace_prefix_debug;

typedef enum {
	LEVEL_NONE,
	LEVEL_DEBUG,
	LEVEL_INFO,
	LEVEL_WARNING,
	LEVEL_ERROR,
	LEVEL_ALL
} LogLevel;

typedef enum
{
	CATEGORY_NONE,
	CATEGORY_INDEXER,
	CATEGORY_QUERY,
	CATEGORY_OPTIMIZER,
	CATEGORY_DICTIONARY,
	CATEGORY_CONTAINER,
	CATEGORY_ALL
} LogCategory;

#ifdef SWIGTCL8
%typemap(in, numinputs = 0) int *majorp, int *minorp, int *patchp %{ $1 = NULL; %}
const char *dbxml_version(int *majorp, int *minorp, int *patchp);
void setLogLevel(LogLevel level, bool enabled);
void setLogCategory(LogCategory category, bool enabled);
#endif /* SWIGTCL8 */

/*
 * All the methods that return pointers to allocated memory.
 * Required so the memory is freed when the objects are deleted.
 */
%newobject XmlContainer::getIndexSpecification(DbTxn*) const;
%newobject XmlContainer::getDocument(DbTxn*, u_int32_t, u_int32_t);
%newobject XmlContainer::parseXPathExpression(DbTxn*, const std::string&, XmlQueryContext*) const;
%newobject XmlContainer::queryWithXPath(DbTxn*, const std::string&, XmlQueryContext*, u_int32_t) const;
%newobject XmlContainer::queryWithXPath(DbTxn *, XmlQueryExpression&, u_int32_t) const;
#if defined(SWIGTCL8) || defined(SWIGPYTHON) || defined(SWIGPHP4)
%newobject XmlContainer::query(DbTxn*, const char*, XmlQueryContext*) const;
#endif

%newobject XmlDocument::queryWithXPath(const std::string&, XmlQueryContext*) const;
%newobject XmlDocument::queryWithXPath(XmlQueryExpression&) const;

%newobject XmlQueryContext::getVariableValue(const std::string&);

#if defined(SWIGJAVA) || defined(SWIGPYTHON)
%newobject XmlResults::next(DbTxn *);
%newobject XmlResults::next();
#endif

#ifdef SWIGTCL8
%newobject XmlValue::asDocument(const XmlQueryContext*) const;
#endif

%newobject XmlIndexSpecification::find(const std::string&, const std::string&);
%newobject XmlIndexSpecification::next();

class XmlContainer
{
public:
	XmlContainer(DbEnv *dbenv, const std::string &name, u_int32_t flags = 0);
	~XmlContainer();

	void setPageSize(u_int32_t pagesize);
	
	void open(DbTxn *txn, u_int32_t flags = 0, int mode = 0);
	bool exists(DbTxn *txn);
	bool isOpen() const;
	void close(u_int32_t flags = 0);
	void setIndexSpecification(DbTxn *txn, const XmlIndexSpecification &index);
	void addIndex(DbTxn *txn, const std::string &uri, const std::string &name, const std::string &index);
	void deleteIndex(DbTxn *txn, const std::string &uri, const std::string &name, const std::string &index);
	void replaceIndex(DbTxn *txn, const std::string &uri, const std::string &name, const std::string &index);
	void upgrade(u_int32_t flags = 0);

%extend {
	/*
	 * Work around a problem in the generated code where objects on the
	 * stack were deleted before they were copied
	 */
	XmlIndexSpecification *getIndexSpecification(DbTxn *txn) const {
		return new XmlIndexSpecification(self->getIndexSpecification(txn));
	}
}

	const std::string &getName() const;
	void setName(const std::string &name);
	u_int32_t putDocument(DbTxn *txn, XmlDocument &document, XmlUpdateContext *context = 0, u_int32_t flags = 0);
	void updateDocument(DbTxn *txn, XmlDocument &document, XmlUpdateContext *context=0);
	
	void remove(DbTxn *txn, u_int32_t flags = 0);
	void rename(DbTxn *txn, const std::string &newName, u_int32_t flags = 0);

#if defined(SWIGTCL8) || !defined(DBXML_USEOVERLOADS)
%name(deleteDocumentByID)
#endif
	void deleteDocument(DbTxn *txn, u_int32_t id, XmlUpdateContext *context = 0, u_int32_t flags = 0);
	void deleteDocument(DbTxn *txn, XmlDocument &document, XmlUpdateContext *context = 0, u_int32_t flags = 0);
	void modifyDocument(DbTxn *txn, XmlModify &modify, XmlUpdateContext *context = 0, u_int32_t flags = 0);

%extend {
	/*
	 * Work around a problem in the generated code where objects on the
	 * stack were deleted before they were copied
	 */
	XmlDocument *getDocument(DbTxn *txn, u_int32_t id, u_int32_t flags = 0) {
		return new XmlDocument(self->getDocument(txn, id, flags));
	}

	XmlQueryExpression *parseXPathExpression(DbTxn *txn, const std::string &query, XmlQueryContext *context = 0) const {
		return new XmlQueryExpression(self->parseXPathExpression(txn, query, context));
	}

	XmlResults *queryWithXPath(DbTxn *txn, const std::string &query, XmlQueryContext *context = 0, u_int32_t flags = 0) const {
		return new XmlResults(self->queryWithXPath(txn, query, context, flags));
	}

#if defined(SWIGTCL8) || !defined(DBXML_USEOVERLOADS)
%name(queryWithXPathExpression)
#endif
	XmlResults *queryWithXPath(DbTxn *txn, XmlQueryExpression &query, u_int32_t flags = 0) const {
		return new XmlResults(self->queryWithXPath(txn, query, flags));
	}

#if defined(SWIGTCL8) || defined(SWIGPYTHON) || defined(SWIGPHP4)
	u_int32_t *query(DbTxn *txn, const char *query, XmlQueryContext *context = 0) const
	{
		// XXX: if context != NULL, should check return type
		XmlResults results = self->queryWithXPath(txn, query, context);
		XmlValue xval;
		
		int num_vals = 0;
		while (results.next(xval))
			++num_vals;
		results.reset();
		u_int32_t *vals = new u_int32_t[num_vals + 1];
		for (int i = 0; i < num_vals; i++) {
			results.next(xval);
			vals[i] = xval.asDocument(context).getID();
		}
		vals[num_vals] = 0;
		return vals;
	}
#endif /* SWIGTCL8 || SWIGPYTHON || SWIGPHP4 */

	void dump(const char *filename, u_int32_t flags = 0) {
		std::ofstream out(filename);
		self->dump(&out, flags);
		out.close();
	}

	void load(const char *filename, u_int32_t flags = 0) {
		std::ifstream in(filename);
		unsigned long lineno = 0;
		self->load(&in, &lineno, flags);
		in.close();
	}

	void verify(const char *filename, u_int32_t flags = 0) {
		std::ofstream out;
		if (flags & DB_SALVAGE)
			out.open(filename);
		self->verify(&out, flags);
		if (flags & DB_SALVAGE)
			out.close();
	}

#ifndef SWIGTCL8
	static void setLogLevel(LogLevel level, bool enabled) {
		DbXml::setLogLevel(level, enabled);
	}

	static void setLogCategory(LogCategory category, bool enabled) {
		DbXml::setLogCategory(category, enabled);
	}

	static int get_version_major() {
		int major;
		(void)dbxml_version(&major, NULL, NULL);
		return major;
	}

	static int get_version_minor() {
		int minor;
		(void)dbxml_version(NULL, &minor, NULL);
		return minor;
	}

	static int get_version_patch() {
		int patch;
		(void)dbxml_version(NULL, NULL, &patch);
		return patch;
	}

	static const char *get_version_string() {
		return dbxml_version(NULL, NULL, NULL);
	}
#endif /* !SWIGTCL8 */
}
};

class XmlDocument
{
public:
	XmlDocument();
	~XmlDocument();

	u_int32_t getID() const;
	void setName(const std::string &name);
	std::string getName() const;
	void setContent(Dbt &content);
	const Dbt *getContent() const;

#if !defined(SWIGTCL8) && !defined(SWIGPYTHON) && !defined(SWIGPHP4)
	/* 
	 * don't trust conversions with non utf-8 encodings 
	 * may be able to enable with some extra code/test
	 * force use of Dbt-based methods
	 */
	void setContent(const std::string &content);
#endif
	void modifyDocument(XmlModify &modify);
%extend {
	std::string getContentAsString() const {
		std::string s;
		return self->getContentAsString(s);
	}
}

	void setMetaData(const std::string &uri, const std::string &prefix, const std::string &name, const XmlValue &value);
	bool getMetaData(const std::string &uri, const std::string &name, XmlValue &value);

/*
	void setMetaData(const std::string &uri, const std::string &prefix, const std::string &name, const Dbt &value);
	bool getMetaData(const std::string &uri, const std::string &name, Dbt &value);
*/

%extend {
	XmlResults *queryWithXPath(const std::string &query, XmlQueryContext *context=0) const {
		return new XmlResults(self->queryWithXPath(query, context));
	}

#if defined(SWIGTCL8) || !defined(DBXML_USEOVERLOADS)
%name(queryWithXPathExpression)
#endif
	XmlResults *queryWithXPath(XmlQueryExpression &query) const {
		return new XmlResults(self->queryWithXPath(query));
	}
}
};

class XmlQueryContext
{
public:
	typedef enum {
		ResultDocuments, ///< Return documents that match the XPath expression.
		ResultValues, ///< Project the XPath expression over the matching document.
		CandidateDocuments, ///< Return documents that may match the XPath expression.
		ResultDocumentsAndValues ///< Return documents and nodes that match the XPath expression.
	} ReturnType;

	typedef enum {
		Eager, ///< Evaluate the whole query immediatly.
		Lazy ///< Evaluate the query as the result set is iterated over.
	} EvaluationType;

	/// Constructor.
	explicit XmlQueryContext(ReturnType returnType = ResultDocuments, EvaluationType evaluationType = Eager);
	~XmlQueryContext();

	void setNamespace(const std::string &prefix, const std::string &uri);
	std::string getNamespace(const std::string &prefix);
	void removeNamespace(const std::string &prefix);
	void clearNamespaces(void);
	void setVariableValue(const std::string &name, const XmlValue &value);
	void setReturnType(ReturnType type = ResultDocuments);
	ReturnType getReturnType() const;
	void setEvaluationType(EvaluationType type = Eager);
	EvaluationType getEvaluationType() const;
	void setWithMetaData(bool withMetaData);
	bool getWithMetaData() const;

%extend {
	XmlValue *getVariableValue(const std::string &name) {
		XmlValue *value = new XmlValue;
		if (self->getVariableValue(name, *value))
			return value;
		else {
			delete value;
			return NULL;
		}
	}
	
#ifdef SWIGTCL8
	const std::string get(const char *name) const {
		XmlValue value;
		return self->getVariableValue(name, value) ? value.asString(self) : "";
	}
	
	void set(const std::string &name, const std::string &value) {
		XmlValue xval(value);
		self->setVariableValue(name, xval);
	}
	
	void setDebugVariable(const std::string &var) {
		self->setVariableValue("dbxml:debug", var);
	}
#endif /* SWIGTCL8 */
}
};

class XmlResults
{
public:
	XmlResults(XmlQueryContext &context, DbTxn *txn = NULL);
	~XmlResults();

#if defined(SWIGJAVA) || defined(SWIGPYTHON) || defined(SWIGPHP4)
%extend {
	XmlValue *next() {
		XmlValue *value = new XmlValue;
		if (self->next(*value))
			return value;
		else {
			delete value;
			return NULL;
		}
	}
}
#else
	bool next(XmlValue &value);
#endif

#if defined(DBXML_USEOVERLOADS)
        /* use overloads if supported */
	bool next(XmlDocument &document);
	bool next(XmlDocument &document, XmlValue &value);
#else
%extend {
	bool nextDocument(XmlDocument &document) {
                return (self->next(document));
        }
}
%extend {
	bool nextDocumentAndValue(XmlDocument &document, XmlValue &value) {
                return (self->next(document, value));
        }
}
#endif

/* next* methods below are compatibility, and deprecated (note: no TCL compat) */
#if defined(SWIGJAVA) || defined(SWIGPYTHON)
%extend {
        XmlValue *next(DbTxn *txn) {
	        XmlValue *value = new XmlValue;
		if (self->next(*value))
			return value;
		else {
			delete value;
			return NULL;
		}
	}
}

%extend {
        bool nextDocument(DbTxn *txn, XmlDocument &document) {
                return (self->next(document));
        }
}
%extend {
        bool nextDocumentAndValue(DbTxn *txn, XmlDocument &document, XmlValue &value) {
                return (self->next(document, value));
        }
}
#endif


#if defined(SWIGPYTHON)
%pythoncode %{
	def __iter__(self): return self
%}
#endif

	void reset();
	size_t size() const;
	void add(const XmlValue &value);
};


class XmlUpdateContext
{
public:
	XmlUpdateContext(XmlContainer &container);
	virtual ~XmlUpdateContext();
};


class XmlValue
{
public:
	typedef enum
	{
		NONE, ///< None
		STRING, ///< String
		NUMBER, ///< Number
		BOOLEAN, ///< Boolean
		DOCUMENT, ///< Document
		NODE, ///< Node
		VARIABLE, ///< Variable
		BINARY ///< Binary
	} Type;

#if	!defined(DBXML_USEOVERLOADS)
        %rename(XmlValueFromDoc) XmlValue(XmlDocument &);
	%rename(XmlValueFromDouble) XmlValue(double);
	%rename(XmlValueFromChar) XmlValue(const  char *);
	%rename(XmlValueFromBool) XmlValue(bool);
	%rename(XmlValueFromNamedType) XmlValue(Type, const  char *);
#endif
	XmlValue();
	XmlValue(XmlDocument &v);
	XmlValue(double v);
	XmlValue(const  char *v);
#ifndef SWIGTCL8
	// Not included in Tcl because there is no way to distinguish between
	// a boolean and a number.
	XmlValue(bool v);
#endif
	XmlValue(Type type, const  char *v);
	~XmlValue();

	Type getType(const XmlQueryContext *context = 0) const;
	bool isNumber(const XmlQueryContext *context = 0) const;
	bool isString(const XmlQueryContext *context = 0) const;
	bool isBoolean(const XmlQueryContext *context = 0) const;
	bool isNode(const XmlQueryContext *context = 0) const;
	bool isDocument(const XmlQueryContext *context = 0) const;
	bool isVariable(const XmlQueryContext *context = 0) const;
	bool isNull() const;

	double asNumber(const XmlQueryContext *context = 0) const;
	std::string asString(const XmlQueryContext *context = 0) const;
	bool asBoolean(const XmlQueryContext *context = 0) const;
#ifndef SWIGTCL8
	XmlDocument asDocument(const XmlQueryContext *context = 0) const;
#else
%extend {
	XmlDocument *asDocument(const XmlQueryContext *context = 0) const {
		return new XmlDocument(self->asDocument(context));
	}
}
#endif
	bool equals(const XmlValue &value, const XmlQueryContext *context = 0) const;
};

class XmlIndexSpecification {
public:
	XmlIndexSpecification();
	virtual ~XmlIndexSpecification();

	void addIndex(const std::string &uri, const std::string &name, const std::string &index);
	void deleteIndex(const std::string &uri, const std::string &name, const std::string &index);
	void replaceIndex(const std::string &uri, const std::string &name, const std::string &index);

	void reset();

%extend {
	XmlIndexDeclaration *find(const std::string &uri, const std::string &name) {
		XmlIndexDeclaration *idecl = new XmlIndexDeclaration(uri, name, "");
		if (self->find(idecl->uri, idecl->name, idecl->index))
			return idecl;
		else {
			delete idecl;
			return NULL;
		}
	}
	
	XmlIndexDeclaration *next() {
		XmlIndexDeclaration *idecl = new XmlIndexDeclaration;
		if (self->next(idecl->uri, idecl->name, idecl->index))
			return idecl;
		else {
			delete idecl;
			return NULL;
		}
	}
}

#if defined(SWIGPYTHON)
%pythoncode %{
	def __iter__(self): return self
%}
#endif
};

#ifndef SWIGJAVA
class XmlIndexDeclaration {
public:
	~XmlIndexDeclaration();

%extend {
	const char *get_uri() const { return self->uri.c_str(); }
	const char *get_name() const { return self->name.c_str(); }
	const char *get_index() const { return self->index.c_str(); }
}
};
#endif

class XmlQueryExpression
{
public:
	XmlQueryExpression(XmlContainer &, XmlQueryContext &);
	~XmlQueryExpression();
	XmlQueryContext &getQueryContext() const;
	const XmlContainer &getContainer() const;

%extend {
	const char *getXPathQuery() const {
		return self->getXPathQuery().c_str();
	}
}
};

class XmlModify
{
public:
	typedef enum {
		InsertAfter,
		InsertBefore,
		Append,
		Update,
		Remove,
		Rename
	} ModificationType;
	typedef enum {
		Element,
		Attribute,
		Text,
		ProcessingInstruction,
		Comment
#if !defined(SWIGPYTHON)
		,None
#endif
	} XmlObject;
	XmlModify(const std::string &xpath, ModificationType operation, XmlObject type,
		  const std::string &name, const std::string &content, int32_t location = -1,
		  XmlQueryContext *context = NULL);
#if defined(DBXML_USEOVERLOADS)
/* TDB: create a static function to enable use of this ctor if !DBXML_OVERLOADS */
	XmlModify(XmlQueryExpression &expression, ModificationType operation, XmlObject type,
		  const std::string &name, const std::string &content, int32_t location = -1);
#endif
	~XmlModify();
	void setNewEncoding(const std::string &newEncoding);
	int getNumModifications();
};
