# global.tcl - global Tcl functions for Setup.
#
# Copyright 1996-1999 Wind River Systems, Inc.
#
# modification history
# --------------------
# 02s,31mar99,wmd  Update the sizes for Setup files in setupSizeGet.
# 02r,29mar99,wmd  Make sure that check for env(SETUP_DEBUG) or env(INF_DEBUG)
#                  are caught.
# 02q,19mar99,wmd  Add proc to write to a file, close the file.
# 02p,03mar99,tcy  Added setupSizeGet() (fix for SPR 25228)
# 02o,24feb99,wmd  Fix so that if exec call returns error, free is still set.
# 02n,02feb99,tcy  modify defGroupSet () to set Tornado2 as default if
#                  folder name selected is a T101 folder
# 02m,28jan99,bjl  added headers and edited for coding conventions.
# 02l,28jan99,wmd  Fix bug in checkPathLen when filename itself is too long.
# 02k,14jan99,wmd  Move procedure debug from INSTALL.TCL to this file.
# 02j,13jan99,wmd  Copy fspace proc from fspace.tcl to this file.
# 02i,18nov98,wmd  added proc checkPathLen
# 02h,16oct98,tcy  added instKeyData[G|S]et()
# 02g,15oct98,tcy  do a unix2dos on filename if win98 host in proc cdFileNameGet
# 02f,13oct98,wmd  do a unix2dos on filename if win95 host in proc cdFileNameGet
# 02e,15sep98,wmd  added cdFileNameGet proc, changed tmpDirGet() to return
#                  /vobs/wpwr/host/<HOST_TYPE>/lib if env var TCL_DEBUG exists.
# 02d,11sep98,tcy  added sysDir[G|S]et ()
# 02c,01sep98,wmd  added destDirDispGet proc.
# 02b,19aug98,wmd  add proc tmpDirGet.
# 02a,22jul98,tcy  moved licenseHomeDirGet/Set () to LICW32.TCL
# 01d,10aug97,pdn  fixed instKeyGet() to trim off white space before return.
# 01c,09feb97,pdn  added exitMsgSet/exitMsgGet functions
# 01b,25nov96,sj   added functions for tempDir, arFlags, currentIndex
# 01a,14nov96,pdn  written
#
# DESCRIPTION
# This file contains the common functions used by the Setup Tcl files.
#
#*/

##############################################################################
#
# userNameGet - returns the name of the user.
# 
# This routine returns the name of the user.  If the user name has not been
# set, an empty string is returned.
#
# SYNOPSIS:
# userNameGet
#
# PARAMETERS: N/A
#
# RETURNS: the user name if it exists, else an empty string.   
# 
# ERRORS: N/A
#

proc userNameGet {} {
    global setupVals

    if {[info exists setupVals(userName)]} {
        return $setupVals(userName)
    } else {
        return ""
    }
}

##############################################################################
#
# userNameSet - sets the name of the user.
# 
# Sets the global variable setupVals(userName) to the name of the user.
#
# SYNOPSIS:
# userNameSet val
#
# PARAMETERS: 
# <val>
# Contains the name of the user. 
#
# RETURNS: N/A
# 
# ERRORS: N/A
#

proc userNameSet {val} {
    global setupVals

    set setupVals(userName) $val
}

##############################################################################
#
# companyNameGet - returns the company name. 
#
# This routine returns the company name.  If the company name has not
# been set, an empty string is returned.
#
# SYNOPSIS:
# companyNameGet
#
# PARAMETERS: N/A
#
# RETURNS: the company name if it exists, else an empty string.
#
# ERRORS: N/A
#

proc companyNameGet {} {
    global setupVals

    if {[info exists setupVals(companyName)]} {
        return $setupVals(companyName)
    } else {
        return ""
    }
}

##############################################################################
#
# companyNameSet - sets the name of the company.
#
# Sets the global variable setupVals(companyName) to the name of the company.
#
# SYNOPSIS:
# companyNameSet val
#
# PARAMETERS:
# <val>
# Contains the name of the company.
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc companyNameSet {val} {
    global setupVals

    set setupVals(companyName) $val
}

##############################################################################
#
# instKeyGet - returns the installation key.
#
# This routine returns the installation key.  If the installation key has not
# been set, an empty string is returned.
#
# SYNOPSIS:
# instKeyGet
#
# PARAMETERS: N/A
#
# RETURNS: the installation key if it exists, else an empty string.
#
# ERRORS: N/A
#

proc instKeyGet {} {
    global setupVals

    if {[info exists setupVals(instKey)]} {
        return [string trim $setupVals(instKey)]
    } else {
        return ""
    }
}

##############################################################################
#
# instKeySet - sets the installation key.
#
# Sets the global variable setupVals(instKey) to the installation key.
#
# SYNOPSIS:
# instKeySet val
#
# PARAMETERS:
# <val>
# Contains the installation key.
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc instKeySet {val} {
    global setupVals

    set setupVals(instKey) $val
}

##############################################################################
#
# instKeyDataGet - returns the installation key data.
#
# This routine returns the installation key data, which includes 
# the installation key history for a particular CD.  If the installation key 
# data has not been set, an empty string is returned.
#
# SYNOPSIS:
# instKeyDataGet
#
# PARAMETERS: N/A
#
# RETURNS: the installation key data if it exists, else an empty string.
#
# ERRORS: N/A
#

proc instKeyDataGet {} {
    global setupVals

    if {[info exists setupVals(keyList)]} {
        return [string trim $setupVals(keyList)] 
    } else {
        return ""
    }
}

##############################################################################
#
# instKeyDataSet - sets the installation key data.
#
# Sets the global variable setupVals(keyList) to the installation key data.
#
# SYNOPSIS:
# instKeyDataSet val
#
# PARAMETERS:
# <val>
# Contains the installation key data.
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc instKeyDataSet {val} {
    global setupVals

    set setupVals(keyList) $val
}

##############################################################################
#
# cdromZipDirGet - returns the directory containing the Setup zip files.
#
# This routine returns the directory containing the Setup zip (WIND) files.  
# If the zip directory has not been set, an empty string is returned.
#
# SYNOPSIS:
# cdromZipDirGet
#
# PARAMETERS: N/A
#
# RETURNS: the zip directory if it exists, else an empty string.
#
# ERRORS: N/A
#

proc cdromZipDirGet {} {
    global setupVals

    if {[info exists setupVals(cdromZipDir)]} {
        return $setupVals(cdromZipDir)
    } else {
        return ""
    }
}

##############################################################################
#
# cdromZipDirSet - sets the zip directory. 
#
# Sets the global variable setupVals(cdromZipDir) to the directory 
# containing the Setup zip (WIND) files.
#
# SYNOPSIS:
# cdromZipDirSet val
#
# PARAMETERS:
# <val>
# Contains the zip directory.
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc cdromZipDirSet {val} {
    global setupVals

    set setupVals(cdromZipDir) $val
}

##############################################################################
#
# cdromRootDirGet - returns the root directory of the CD-ROM.
#
# This routine returns the root directory of the CD-ROM, or the root
# of the Setup tree.  If the root directory has not been set, an empty 
# string is returned.
#
# SYNOPSIS:
# cdromRootDirGet
#
# PARAMETERS: N/A
#
# RETURNS: the Setup root directory if it exists, else an empty string.
#
# ERRORS: N/A
#

proc cdromRootDirGet {} {
    global env

    if {[info exists env(CD_ROOT)]} {
        return $env(CD_ROOT)
    } else {
        return ""
    }
}

##############################################################################
#
# cdFileNameGet - returns the root directory of the CD-ROM.
#
# This routine returns the root directory of the CD-ROM, or the root
# of the Setup tree.  If the root directory has not been set, an empty 
# string is returned.
#
# SYNOPSIS:
# cdromRootDirGet
#
# PARAMETERS: N/A
#
# RETURNS: the Setup root directory if it exists, else an empty string.
#
# ERRORS: N/A
#

proc cdFileNameGet {fileName} {
    global ctrlVals
    
    # return only valid all uppercase or all lower case path/fileName

    set fn [string toupper $fileName]
    if {$ctrlVals(version) == "WIN95" || $ctrlVals(version) == "WIN98"} {
        set fn [unixToDos [file join [cdromRootDirGet] $fn]]
    } else {
        set fn [file join [cdromRootDirGet] $fn]
    }
    if {[file exists $fn]} {
        return $fn
    } 

    set fn [string tolower $fileName]
    if {$ctrlVals(version) == "WIN95" || $ctrlVals(version) == "WIN98"} {
        set fn [unixToDos [file join [cdromRootDirGet] $fn]]
    } else {
        set fn [file join [cdromRootDirGet] $fn]
    }
    if {[file exists $fn]} {
        return $fn
    }

    if {$ctrlVals(version) == "WIN95" || $ctrlVals(version) == "WIN98"} {
        set fn [unixToDos [file join [cdromRootDirGet] $fileName]]
    } else {
        set fn [file join [cdromRootDirGet] $fileName]
    }
    return $fn
}

##############################################################################
#
# tmpDirGet - returns the temporary directory.
#
# This routine returns the temporary directory set by the environment
# variable TMP.  For debugging purposes and to use the tcl debugger, setting 
# the environment variable TCL_DEBUG will return the [windHostType]/lib 
# directory in the current view.  If the temp directory has not been set, 
# an empty string is returned.
#
# SYNOPSIS:
# tmpDirGet
#
# PARAMETERS: N/A
#
# RETURNS: /vobs/wpwr/host/[windHostTypeGet] if env(TCL_DEBUG) exists,
#          the value set by the environment variable TMP, or the empty
#          string if neither exists.
#
# ERRORS: N/A
#

proc tmpDirGet {} {
    global env

    if {[info exists env(TCL_DEBUG)]} {
        return /vobs/wpwr/host/[windHostTypeGet]/lib
    }

    if {[info exists env(TMP)]} {
        return $env(TMP)
    } else {
        return ""
    }
}

##############################################################################
#
# uninstHomeDirGet - returns the uninstall home directory.
#
# This routine returns the uninstall home directory set by 
# setupVals(uninstHomeDir).  If the uninstall home directory has not
# been set, an empty string is returned.
#
# SYNOPSIS:
# uninstHomeDirGet
#
# PARAMETERS: N/A
#
# RETURNS: the installation key if it exists, else an empty string.
#
# ERRORS: N/A
#

proc uninstHomeDirGet {} {
    global setupVals

    if {[info exists setupVals(uninstHomeDir)]} {
        return $setupVals(uninstHomeDir)
    } else {
        return ""
    }
}

##############################################################################
#
# uninstHomeDirSet - sets the uninstall home directory.
#
# Sets the global variable setupVals(uninstHomeDir) to the uninstall
# home directory.
#
# SYNOPSIS:
# uninstHomeDirSet val
#
# PARAMETERS:
# <val>
# Contains the uninstall home directory.  
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc uninstHomeDirSet {val} {
    global setupVals

    set setupVals(uninstHomeDir) $val
}

##############################################################################
#
# destDirDispGet - returns the destination directory with slashes displayed
#                  in Unix format.
#
# Returns the destination directory to be displayed to the user.  Slashes 
# are converted to Unix format.
# 
# SYNOPSIS:
# destDirDispGet 
#
# PARAMETERS: N/A
#
# RETURNS: the destination directory with slashes converted to Unix format.
#
# ERRORS: N/A
#

proc destDirDispGet {} {

    set destDir [destDirGet]
    if {$destDir != ""} {
        set destDir [dosToUnix $destDir]
    }
    return $destDir
}

##############################################################################
#
# destDirGet - returns the destination directory.
#
# This routine returns the destination directory location to install
# the files for the product(s).  If the destination directory has not
# been set, an empty string is returned.
#
# SYNOPSIS:
# destDirGet
#
# PARAMETERS: N/A
#
# RETURNS: the destination directory if it exists, else an empty string.
#
# ERRORS: N/A
#

proc destDirGet {} {
    global setupVals

    if {[info exists setupVals(destDir)]} {
        return $setupVals(destDir)
    } else {
        return ""
    }
}

##############################################################################
#
# destDirSet - sets the destination directory.  
#
# Sets the global variable setupVals(destDir) to the destination directory.
#
# SYNOPSIS:
# destDirSet val
#
# PARAMETERS:
# <val>
# Contains the destination directory.
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc destDirSet {val} {
    global setupVals

    set setupVals(destDir) $val
}

##############################################################################
#
# sysDirGet - returns the Windows system directory.
#
# This routine returns the Windows system directory.  If the system directory 
# has not been set, an empty string is returned.
#
# SYNOPSIS:
# sysDirGet
#
# PARAMETERS: N/A
#
# RETURNS: the system directory if it exists, else an empty string.
#
# ERRORS: N/A
#

proc sysDirGet {} {
    global setupVals

    if {[info exists setupVals(sysDir)]} {
        return $setupVals(sysDir)
    } else {
        return ""
    }
}

##############################################################################
#
# sysDirSet - sets the system directory.  
#
# Sets the global variable setupVals(sysDir) to the Windows system directory.
#
# SYNOPSIS:
# sysDirSet val
#
# PARAMETERS:
# <val>
# Contains the system directory.
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc sysDirSet {val} {
    global setupVals

    set setupVals(sysDir) $val
}

##############################################################################
#
# cdromBinDirGet - returns the host-specific CD-ROM executable files.
#
# This routine returns the CD-ROM directory containing the host-specific
# binary and executable files, for example env(CD_ROOT)/SUN4/SOLARIS2 or
# env(CD_ROOT)\X86\WIN32. If the bin directory has not been set, 
# an empty string is returned.  
#
# SYNOPSIS:
# cdromBinDirGet
#
# PARAMETERS: N/A
#
# RETURNS: the bin directory if it exists, else an empty string.
#
# ERRORS: N/A
#

proc cdromBinDirGet {} {
    global env

    if {[info exists env(CD_BIN)]} {
        return $env(CD_BIN)
    } else {
        return ""
    }
}

##############################################################################
#
# windHostTypeGet - returns the WIND_HOST_TYPE.
#
# This routine returns the WIND_HOST_TYPE set by the global variable 
# setupVals(windHostType).  If the WIND_HOST_TYPE has not been set, 
# an empty string is returned.  
#
# SYNOPSIS:
# windHostTypeGet
#
# PARAMETERS: N/A
#
# RETURNS: the WIND_HOST_TYPE if it exists, else an empty string.
#
# ERRORS: N/A
#

proc windHostTypeGet {} {
    global setupVals

    if {[info exists setupVals(windHostType)]} {
        return $setupVals(windHostType)
    } else {
        return ""
    }
}

##############################################################################
#
# windHostTypeSet - sets the WIND_HOST_TYPE.
#
# Sets the global variable setupVals(windHostType) to the WIND_HOST_TYPE.
#
# SYNOPSIS:
# windHostTypeSet val
#
# PARAMETERS:
# <val>
# Contains the WIND_HOST_TYPE [x86-win32 | sun4-solaris2 | parisc-hpux10].
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc windHostTypeSet {val} {
    global setupVals

    set setupVals(windHostType) $val
}

##############################################################################
#
# lastErrorGet - returns the last setup error.  
#
# This routine returns the last setup error set by the global variable
# setupVals(lastError).  If the last error has not been set, an empty string 
# is returned.  
#
# SYNOPSIS:
# lastErrorGet
# 
# PARAMETERS: N/A
#
# RETURNS: the last error if it exists, else an empty string.  
#
# ERRORS: N/A
#

proc lastErrorGet {} {
    global setupVals

    if {[info exists setupVals(lastError)]} {
        return $setupVals(lastError)
    } else {
        return ""
    }
}

##############################################################################
#
# lastErrorSet - sets the last setup error and increments the error count.
#
# Sets the global variable setupVals(lastError) to the last setup error
# and increments the value of setupVals(errorCount).
#
# SYNOPSIS:
# lastErrorSet val
#
# PARAMETERS:
# <val>
# Contains the last setup error.
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc lastErrorSet {val} {
    global setupVals

    set setupVals(lastError) $val
    incr setupVals(errorCount)
}

##############################################################################
#
# lastErrorMsgSet - sets the last setup error.
#
# Sets the global variable setupVals(lastError) to the last setup error.
# Note that unlike lastErrorSet the errorCount is not incremented.
#
# SYNOPSIS:
# lastErrorMsgSet val
#
# PARAMETERS:
# <val>
# Contains the last setup error.
# RETURNS: N/A
#
# ERRORS: N/A
#

proc lastErrorMsgSet {val} {
    global setupVals

    set setupVals(lastError) $val
}

##############################################################################
#
# errorCountGet - returns the error count.
#
# This routine returns the error count recorded by the global variable
# setupVals(errorCount).  
#
# SYNOPSIS:
# errorCountGet
# 
# PARAMETERS: N/A
#
# RETURNS: the error count.  
#
# ERRORS: N/A
#

proc errorCountGet {} {
    global setupVals

    if {[info exists setupVals(errorCount)]} {
        return $setupVals(errorCount)
    } else {
        return 0
    }
}

##############################################################################
#
# cdromDescGet - returns the CD-ROM description.
#
# This routine returns the CD-ROM description string set by the global
# variable setupVals(cdDesc).  If the description has not been set, an empty 
# string is returned.  
#
# SYNOPSIS:
# cdromDescGet
# 
# PARAMETERS: N/A
#
# RETURNS: the CD-ROM description string, else 0 if it does not exist.
#
# ERRORS: N/A
#

proc cdromDescGet {} {
    global setupVals

    if {[info exists setupVals(cdDesc)]} {
        return $setupVals(cdDesc)
    } else {
        return 0
    }
}

##############################################################################
#
# cdromDescSet - sets the CD-ROM description.
#
# Sets the global variable setupVals(cdDesc) to the CD-ROM description.
# 
# SYNOPSIS:
# cdromDescSet val
#
# PARAMETERS:
# <val>
# Contains the CD-ROM description.
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc cdromDescSet {val} {
    global setupVals 

    set setupVals(cdDesc) $val
}

##############################################################################
#
# arFlagsSet - sets the arflags for the current product index.
#
# Sets the global variable arFlags([currentIndexGet]) for the current
# product index.  
# 
# SYNOPSIS:
# arFlagsSet val
#
# PARAMETERS:
# <flags>
# Contains the arflags for the current product index.
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc arFlagsSet {flags} {
    global arFlags 

    set arFlags([currentIndexGet]) $flags
}

##############################################################################
#
# arFlagsGet - returns the arFlags given the product index number.
#
# This routine returns the arFlags for the specified index based on the
# value of the global array arFlags($index).  If no value has been set
# the default flags "-cru" are returned.
# 
#
# SYNOPSIS:
# cdromDescGet
# 
# PARAMETERS: 
# <index>
# The product index number.  
#
# RETURNS: the arflags for the specified product index, otherwise
#          the default "-cru" if no arflags have been specified.
#
# ERRORS: N/A
#

proc arFlagsGet {index} {
    global arFlags

    if ![info exists arFlags($index)] {
        set arFlags($index) "-cru"
    }
    return $arFlags($index)
}

##############################################################################
#
# currentIndexSet - sets the current product index.
#
# Sets the global variable currentIndex to the specified product
# index.  
# 
# SYNOPSIS:
# currentIndexSet index
#
# PARAMETERS:
# <index>
# The current product index.
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc currentIndexSet {index} {
    global currentIndex

    set currentIndex $index
}

##############################################################################
#
# currentIndexGet - returns the current product index.
#
# This routine returns the current product index set by the global 
# variable currentIndex.  If no value has been set an empty string is
# returned.
#
# SYNOPSIS:
# currentIndexGet
# 
# PARAMETERS: N/A
#
# RETURNS: the current product index if it exists, else an empty string.
#
# ERRORS: N/A
#

proc currentIndexGet {} {
    global currentIndex

    if {[info exists currentIndex]} {
        return $currentIndex
    } else {
        return ""
    }
}

##############################################################################
#
# tempDirGet - returns the temporary directory.
#
# This routine returns the temporary directory set by the environment
# variable TMP.  
#
# SYNOPSIS:
# tempDirGet
# 
# PARAMETERS: N/A
#
# RETURNS: the temporary directory.
#
# ERRORS: 
# "Temp dir not set"
#

proc tempDirGet {} {
    global env

    if {[info exists env(TMP)]} {
        return $env(TMP)
    } else {
        error "Temp dir not set" 
    }
}

##############################################################################
#
# exitMsgSet - sets the exit message.
#
# Sets the global variable exitMessage to the specified exit message.
# 
# SYNOPSIS:
# currentIndexSet msg
#
# PARAMETERS:
# <msg>
# The exit message.
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc exitMsgSet {msg} {
    global exitMessage

    set exitMessage $msg
}

##############################################################################
#
# exitMsgGet - returns the exit message.  
#
# This routine returns the exit message.  If it does not exist, it returns
# the message "Installation complete."
#
# SYNOPSIS:
# exitMsgGet
# 
# PARAMETERS: N/A
#
# RETURNS: the exit message if it exists, else "Installation complete."
#
# ERRORS: N/A
#

proc exitMsgGet {} {
    global exitMessage

    if {![info exists exitMessage]} {
        set exitMessage "Installation complete."
    }

    return $exitMessage
}

##############################################################################
#
# defGroupSet - sets the default group.
#
# Sets the global variable defGroup to the specified default group. 
#
# SYNOPSIS:
# defGroupSet group
#
# PARAMETERS:
# <group>
# The default group.
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc defGroupSet {group} {
    global defGroup

    if {[portMapperIconExist $group] == 1} {
        set defGroup "Tornado2"
    } else {
        set defGroup $group
    }
}

##############################################################################
#
# defGroupGet - returns the default group.
#
# This routine returns the default group.  If it does not exist an empty
# string is returned.
#
# SYNOPSIS:
# defGroupGet
#
# PARAMETERS: N/A
#
# RETURNS: the default group if it exists, else an empty string.
#
# ERRORS: N/A
#

proc defGroupGet {} {
    global defGroup

    if ![info exists defGroup] {
        set defGroup ""
    }

    return $defGroup
}

##############################################################################
#
# setupSizeGet - returns the size of setup directory
#
# This routine returns the size of setup directory
#
# SYNOPSIS:
# setupSizeGet
#
# PARAMETERS: N/A
#
# RETURNS:  the size of setup directory
#
# ERRORS: N/A
#

proc setupSizeGet {} {

    switch [windHostTypeGet] {
        x86-win32 {
            return 6.3
        }
        sun4-solaris2 {
            return 7.2
        }
        parisc-hpux10 {
            return 11.3
        }
        default {
            return 6.3
        }
    }
}

##############################################################################
#
# instTypeSet - sets the installation type.
#
# Sets the global variable instType to the installation type (for example,
# "icon" for an icon-only installation).
#
# SYNOPSIS:
# instTypeSet type
#
# PARAMETERS:
# <type>
# The installation type.
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc instTypeSet {type} {
    global instType

    set instType $type
}

##############################################################################
#
# instTypeGet - returns the installation type.
#
# This routine returns the installation type (for example, "icon" for 
# an icon-only installation).  If the installation type has not been
# set, an empty string is returned.
#
# SYNOPSIS:
# userNameGet
#
# PARAMETERS: N/A
#
# RETURNS: the installation type if it exists, else an empty string.
#
# ERRORS: N/A
#

proc instTypeGet {} {
    global instType

    if ![info exists instType] {
        set instType ""
    }

    return $instType
}

##############################################################################
#
# checkPathLen - checks the path length and trims the path if necessary.
#                
# This routine checks the length of a given path.  If the length is
# over 50 characters, part of the path is replaced with "..."  This
# allows a long path to fit in a dialog window.
#
# SYNOPSIS:
# checkPathLen path
#
# PARAMETERS: 
# <path>
# A directory path.
#
# RETURNS: The original path partially replaced with "..." if over
#          50 characters.  
#
# ERRORS: N/A
#

proc checkPathLen {path} {

    if {[string length $path] >= 50} {
        set totLen [string length $path]
        set lastIndex [string last "/" $path]
        if {$lastIndex > [string last "\\" $path]} {
            # Unix type path

            set path2 [string range $path 0 [expr $lastIndex - 1]]
            set fname [string range $path [expr $lastIndex + 1] $totLen]
            set lastIndex2 [string last "/" $path2]
            while {[expr [string length $path2] + [string length $fname] + 5] \
                >= 50} {
                set path2 [string range $path2 0 [expr $lastIndex2 -1]]
                set lastIndex2 [string last "/" $path2]
                if {$lastIndex2 == -1} {
                    break;
                }
            }
           set path [format "%s/.../%s" $path2 $fname]

        } else { 
            # DOS type path

            set lastIndex [string last "\\" $path]
            set path2 [string range $path 0 [expr $lastIndex - 1]]
            set fname [string range $path [expr $lastIndex + 1] $totLen]
            set lastIndex2 [string last "\\" $path2]
            while {[expr [string length path2] + [string length $fname] + 5] \
                >= 50} {
                set path2 [string range $path2 0 [expr $lastIndex2 -1]]
                set lastIndex2 [string last "\\" $path2]
                if {$lastIndex2 == -1} {
                    break;
                }
            }
            set $path [format "%s\\...\\%s" $path2 $fname]

        }
    }
    return $path
}

#############################################################################
#
# fspace - returns free space available on Unix hosts
#
# This procedure returns the amount of free space avaiable on the given drive.
#
# SYNOPSIS:
# fspace dir
#
# PARAMETERS: 
# <dir>
# a directory path.
#
# RETURNS: the number of free space in kilobytes avaiable
#
# ERRORS: N/A
#

proc fspace {dir} {

    if {![file isdirectory $dir]} {
        return "$dir: bad directory name"
    }

    # get the directory name in extension

    set here [pwd]
    cd $dir
    set dir [pwd]

    set free "unknown"

    switch [windHostTypeGet] {
        sun4-solaris2 {
            if {![catch {exec /bin/df -k $dir} result]} {
                set free [lindex $result 10]
            } else {
                set free [lindex $result 10]
            }
        }

        parisc-hpux10 {
            set found 0
            set ix 0
            while {$found == 0} {
                incr ix
                if {$ix > 30} {
                    break
                }
                if {[catch "exec /bin/df $dir" res]} {
                    # go backward one step looking for actual mounting
                    # point or device name alias
                    set dir [file dirname $dir]
                } else {
                    set freeSize [lindex [exec /bin/df -k $dir | /bin/sed -e "/total/d" -e "/used/d" ] 0]
                    if {[regexp {[^0-9]+} $freeSize] == 0} {
                        set free $freeSize
                    }
                    set found 1
                }
            }
        }

        default         {}
    }
    cd $here
    return $free
}

##############################################################################
#
# debug - turns on debug mode
#
# SYNOPSIS
# debug
#
# PARAMETERS: N/A
#
# RETURNS: true if the environment var SETUP_DEBUG exists, otherwise false
#
# ERRORS: N/A
#

proc debug {} {
    global env
    global setupVals

    if {[info exists env(SETUP_DEBUG)]} {
        return 1
    } else {
        return 0
    }
}

##############################################################################
#
# openSetupDebugLog - open the setup log file
#
# SYNOPSIS
# openSetupDeubgLog
#
# PARAMETERS: N/A
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc openSetupDebugLog {} {
    global env setupVals

    if {[info exists env(SETUP_DEBUG)] && $env(SETUP_DEBUG) != 0} { 
        if {[info exists env(SETUP_DBGLOGFILE)]} {
            if {![info exists setupVals(DBGLOG_FD)]} {
                if {[catch {open $env(SETUP_DBGLOGFILE) "w"} setupVals(DBGLOG_FD)]} {
                    puts "Can't open $env(SETUP_DBGLOGFILE)"
                }
            }
        }
    }
    if {[info exists env(INF_DEBUG)] &&  $env(INF_DEBUG) != 0} {
        if {[info exists env(SETUP_DBGLOGFILE)]} {
            if {![info exists setupVals(DBGLOG_FD)]} {
                if {[catch {open $env(SETUP_DBGLOGFILE) "w"} setupVals(DBGLOG_FD)]} {
                    puts "Can't open $env(SETUP_DBGLOGFILE)"
                }
            }
        }
    }
}

##############################################################################
#
# closeSetupDebugLog - close the setup log file
#
# SYNOPSIS
# closeSetupLog
#
# PARAMETERS: N/A
#
# RETURNS: N/A
#
# ERRORS: N/A
#

proc closeSetupDebugLog {} {
    global env
    global setupVals

    if {[info exists setupVals(DBGLOG_FD)]} {
        catch {close $setupVals(DBGLOG_FD)}
    }
}

#############################################################################
#
# dbgputs - wrapper for debug puts function.
#
# Wrapper for the puts function.  Only prints out the specified string
# either to the setup debug log file in env(SETUP_DBGLOGFILE) or the console,
# if the environment variable SETUP_DEBUG exists and is set to a nonzero value. 
#
# SYNOPSIS
# dbgputs <line>
#
# PARAMETERS: 
#   line : string to output.   
#
# RETURNS: N/A
#   
# ERRORS: N/A
#

proc dbgputs {line} {
    global env
    global setupVals

    if {[info exists env(SETUP_DEBUG)]} {
    	if {$env(SETUP_DEBUG) != 0 && [info exists setupVals(DBGLOG_FD)]} {
            puts $setupVals(DBGLOG_FD) $line
            flush $setupVals(DBGLOG_FD)
        } else {
            puts $line
        }
    }
}
